#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
""" Reset an OLT device.

This MCMS REST API example script resets an ONU device.

Example:

  ./reset_onu.py --url https://10.2.10.29/api --user <email> --password <password> --onu ALPHe30cadcf


"""

import argparse
from api_client import ApiClient


def main():
    """ Entry point for the script. """
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("-d", "--db", action="store", dest="database", default="Default", required=False, help="Name of the database.")
    parser.add_argument("-h", "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("--olt", action="store", dest="olt", default=None, required=True, help="OLT MAC Address (e.g., 70:b3:d5:52:37:24)")
    parser.add_argument("-l", "--url", action="store", dest="url", default="https://10.2.10.29/api", required=False, help="URL of the MCMS API server (e.g., https://10.2.10.29/api).")
    parser.add_argument("-p", "--password", action="store", dest="password", default="tibit", required=False, help="User password to authenticate with.")
    parser.add_argument("-u", "--user", action="store", dest="user", default="tibit@tibitcom.com", required=False, help="User email to authenticate with.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    # Instantiate an API Client Connection
    api_client = ApiClient(args.url, args.verbose)

    # Login to the web server
    api_client.login(args.user, args.password)

    # Select the database to use for this session
    api_client.select_database(args.database)

    # Reset the OLT
    status, response_data = api_client.request("PUT", f"/v1/olts/{args.olt}/reset/")

    # Logout of the web server to terminate the session
    api_client.logout()

    # Display the ONU registration status of for the OLT
    if status == 200:
        print(f"\nSuccessfully reset OLT {args.olt}")
    else:
        print(f"\nFailed to reset OLT {args.olt}")

if __name__ == '__main__':
    main()
