"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""
import json, os

from drf_spectacular.utils import inline_serializer
from rest_framework import serializers
from rest_framework.fields import CharField
from api.settings import IN_PRODUCTION, PONMGR_OPT_PATH
from version import VERSION

from docs.json_reference_resolver import PRODUCTION


# ==================================================
# ============== Response Serializers ==============
# ==================================================

class OkResponseSerializer(serializers.Serializer):
    """ Serializer for 200 status API responses """
    status = serializers.ChoiceField(default="success", choices=["success", "validation warning"])
    details = serializers.DictField(required=False)
    data = serializers.DictField(required=False)

class CreatedResponseSerializer(serializers.Serializer):
    """ Serializer for 201 status API responses """
    status = serializers.ChoiceField(default="success", choices=["success", "validation warning"])
    details = serializers.DictField(required=False)


class ErrorResponseSerializer(serializers.Serializer):
    """ Serializer for 400-599 status API Responses """
    status = serializers.CharField(default="fail")
    details = serializers.DictField()


# ==================================================
# ============== Request Serializers ===============
# ==================================================

def RequestSerializer(name: str, data_fields: dict):
    """ Returns a serializer with the given fields nested in the 'data' field """
    return inline_serializer(name=name, fields={
        "data": inline_serializer(name=name + "Data", fields=data_fields)
    })


class CharFieldSerializer(serializers.Serializer):
    """ Payload serializer for requests where data is a string """
    data = CharField()


# ==================================================
# ================ Swagger Helpers =================
# ==================================================

def schema(schema_collection):
    with open(schema_collection, 'r') as file:
        schema_collection = json.load(file)
    return schema_collection


def get_schema(collection):
    """ Returns the location of the swagger friendly schema file"""
    # formats version
    version = VERSION
    if "rc" in version or "ma" in version:
        strsplit = version.split("-")
        version = strsplit[0]

    # gets swagger schema if live docs is enabled
    if IN_PRODUCTION:
        schema_path = f"{PONMGR_OPT_PATH}/api/docs/swagger_schema"
        path = "swagger-" + collection + ".json"
    else:
        schema_path = 'docs/swagger_schema'
        path = "swagger-" + collection + ".json"

    schema_file = os.path.join(schema_path, path)

    try:
        # tries to load the swagger schema file
        schema(schema_file)
    except FileNotFoundError:
        schema_path = "schema_files/" + version
        path = collection + ".json"
        schema_file = os.path.join(schema_path, path)
    return schema_file


def get_unversioned_schema(collection):
    """ Returns the location of the unversioned swagger friendly schema file"""

    # gets swagger schema if live docs is enabled
    if IN_PRODUCTION:
        schema_path = f"{PONMGR_OPT_PATH}/api/docs/swagger_schema"
        path = "swagger-" + collection + ".json"
    else:
        schema_path = "schema_files/unversioned_schema_files"
        path = collection + ".json"

    schema_file = os.path.join(schema_path, path)
    return schema_file

