"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

from ipaddress import IPv6Address, ip_address
from typing import Any, Union
from urllib.parse import quote_plus
from utils.keyring_helper import get_mongo_password

def lookup_value_from_dict(config: dict, key) -> Any:
    """
        Validates if a key is present in a dictionary and returns its value, else None
    """
    if key in config.keys():
        return config[key]
    return None

def get_mongo_uri(config: dict) -> Union[None, str]:
    """
    Generates URI compatible for connection to mongoengine based on values present in user_database.json
    Args:
        config: dictionary containing information on mongo DB and authentication

    Returns:
        Generated URI as a string
    """
    mongo_db_uri = lookup_value_from_dict(config, "db_uri")

    if mongo_db_uri:
        return mongo_db_uri

    mongo_name = lookup_value_from_dict(config, "name")
    mongo_host = lookup_value_from_dict(config, "host")
    mongo_port = lookup_value_from_dict(config, "port")
    mongo_user_name = lookup_value_from_dict(config, "username")
    mongo_password = get_mongo_password(config)
    mongo_auth_db = lookup_value_from_dict(config, "auth_db")
    mongo_auth_enabled = lookup_value_from_dict(config, "auth_enable")
    mongo_tls_enabled = lookup_value_from_dict(config, "tls_enable")
    mongo_ca_cert_path = lookup_value_from_dict(config, "ca_cert_path")

    failed = None
    if None in (mongo_name, mongo_host, mongo_port):
        return failed

    if mongo_auth_enabled and None in (mongo_user_name, mongo_password, mongo_auth_db):
        return failed

    if mongo_tls_enabled and mongo_ca_cert_path is None:
        return failed

    options_tokens = []
    mongo_srv_scheme = lookup_value_from_dict(config, "dns_srv")
    mongo_replica_set_enable = lookup_value_from_dict(config, "replica_set_enable")
    mongo_replica_set_name = lookup_value_from_dict(config, "replica_set_name")
    mongo_replica_set_hosts = lookup_value_from_dict(config, "replica_set_hosts")

    # if not using dns_srv and not using a replica set, use "host" and "port"
    if not mongo_srv_scheme:
        scheme_token = "mongodb"

        if not mongo_replica_set_enable:
            try:
                is_ipv6 = isinstance(ip_address(mongo_host), IPv6Address)
            except ValueError:
                is_ipv6 = False

            if is_ipv6:
                connect_token = f"[{mongo_host}]"
            else:
                connect_token = f"{mongo_host}"

            # Add port
            connect_token += f":{mongo_port}"

        else:
            if mongo_replica_set_hosts not in (None, []):
                connect_token = ",".join(mongo_replica_set_hosts)
            else:
                return failed

            if mongo_replica_set_name not in (None, ""):
                options_tokens.append(f'replicaSet={mongo_replica_set_name}')
            else:
                return failed
    else:
        scheme_token = "mongodb+srv"

        # if dns_srv and not using a replica_set, use "host" but not "port"
        connect_token = f"{mongo_host}"

        if mongo_replica_set_enable:
            # if using dns_srv and using replica_set, use "host", don't use "port" and set "replicaSet" option
            if mongo_replica_set_name not in (None, ""):
                options_tokens.append(f'replicaSet={mongo_replica_set_name}')
            else:
                return failed

    auth_token = ""
    if mongo_auth_enabled:
        # Percent-encode usernames and passwords to handle special characters inside of them
        auth_token = f"{quote_plus(mongo_user_name)}:{quote_plus(mongo_password)}@"
        options_tokens.append(f"authSource={mongo_auth_db}")

    options_tokens.append("tls=" + str(mongo_tls_enabled).lower())

    if mongo_tls_enabled:
        options_tokens.append(f"tlsCAFile={mongo_ca_cert_path}")

    # directConnection flag is set to 'true' when replicaSets are disabled for the MongoDB connection
    options_tokens.append("directConnection={}".format("false" if mongo_replica_set_enable else "true"))

    db_token = mongo_name
    _options_string = "&".join(options_tokens)
    db_uri = f'{scheme_token}://{auth_token}{connect_token}/{db_token}?{_options_string}'

    return db_uri