#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#

''' JSON Schema Validation Utility.

The JSON Schema Validation Utility is a library that utilizes the Python
jsonschema package for valdating JSON data against a JSON Schema as defined
by Draft 7 (https://json-schema.org/specification-links.html#draft-7).

JSON Schema files are defined for each PON Controller MongoDB collection
(ONU-CFG, ONU-STATE, etc.) and are located in the
/opt/tibit/ponmgr/api/schema_files directory.

The COMMON-TYPES.json schema is a special schema with common type
definitions that are shared accross all MongoDB schemas.

'''

import json
import os
from typing import Any, Optional, Tuple
import re
import pathlib
from bson.json_util import dumps
import jsonschema


# Override the jsonschema 'integer' type checker to require an python integer data type
def is_int(checker, instance):
    """ Override the 'integer' type checker to verify the value is an instance of int. """
    return jsonschema.Draft7Validator.TYPE_CHECKER.is_type(instance, "integer") and isinstance(instance, int)

# Override the 'required' validator to allow for skipping required checks (e.g., PATCH)
class RequiredValidationError(jsonschema.exceptions._Error):
    """ Custom validation error exception for catching missing 'required' fields errors. """

def required_validator(validator, required, instance, schema):
    """ Override the 'required' value to return a custom validation error exception. """
    if not validator.is_type(instance, "object"):
        return
    for property_ in required:
        if property_ not in instance:
            yield RequiredValidationError("%r is a required property" % property_)

def natural_sort_key(key):
    """ Key for sorting a list in the way that humans expect. """
    convert = lambda text: int(text) if text.isdigit() else text
    return [convert(c) for c in re.split('([0-9]+)', key)]

class JsonSchemaValidator:
    """ JSON Schema Validator

    The JSON Schema Validator class implements JSON schema validatation
    and is used for validating the data payloads of POST and PUT requests
    for the PON Manager REST API.

    """
    def __init__(self, schema_path: str):
        """ JSON Schema Validator constructor.

        The JSON Schemas are organized such that there is a .json schema file
        for each PON Controller MongoDB collection. This method loads .json
        schema files from the specified directory and instantiates a jsonschema
        validator for each schema file. There is a .json schema file for each
        PON Controller MongoDB collection.

        Args:
            schema_path: The path to the directory containing JSON schema files.
        """
        self.schema_versions = []
        self.current_schema_version = None
        self.schemas = {}
        self.collections = {}
        file_name = ""
        collection = ""

        try:
            # Load schema versions from the schema directory
            for schema_version in os.listdir(schema_path):
                schema_version_path = f"{schema_path}/{schema_version}"
                if os.path.isdir(schema_version_path) and self._is_schema_version_dir(schema_version):
                    self.schema_versions.append(schema_version)

            self.current_schema_version = max(self.schema_versions, key=natural_sort_key, default=None)

            # Load schema files for each schema version
            for schema_version in self.schema_versions:
                schemas = {}
                schema_version_path = f"{schema_path}/{schema_version}"
                for file_name in os.listdir(schema_version_path):
                    file_path = f"{schema_version_path}/{file_name}"
                    if os.path.isfile(file_path) and file_name.endswith(".json"):
                        with open(file_path, 'r', encoding="utf-8") as file:
                            schema = json.load(file)
                            if 'title' in schema:
                                schemas[schema['title']] = schema

                # Check schema files for errors
                for collection, schema in schemas.items():
                    jsonschema.Draft7Validator.check_schema(schema)

                # Store the schemas for this version
                self.schemas[schema_version] = schemas

                # Get a list of collection names with validators
                self.collections[schema_version] = schemas.keys()

            # Create a schema and collection entries for the "current" version
            self.schemas["current"] = self.schemas[self.current_schema_version]
            self.collections["current"] = self.collections[self.current_schema_version]

        except json.JSONDecodeError as err:
            print(f"ERROR: JSON schema file format/lint failed for {file_name}.")
            print(f"ERROR: {err}")
        except jsonschema.SchemaError as err:
            print(f"ERROR: JSON schema file validation failed for {collection}.")
            print(f"ERROR: {err}")
        except jsonschema.ValidationError as err:
            print(f"ERROR: Failed to instantiate JSON schema validator for {collection}.")
            print(f"ERROR: {err}")
        except KeyError as err:
            print("ERROR: Failed to instantiate JSON schema validators.")
            print(f"ERROR: {err}")


    def validate(self, collection_name: str, doc: str, validate_required: bool = True, strict_validation: bool = False, schema_version: str="current", schema: dict=None) -> Tuple[bool, dict]:
        """
        Validate a document against the JSON schema defined for the specified
        MongoDB collection.

        Args:
            collection_name: The name of the collection in MongoDB (e.g., ONU-CFG).
            doc: The document to perform JSON schema validation on.
            validate_required: Check for missing 'required' fields.
            strict_validation(optional): Defaults to false. When true, returns error if a required field is missing
            schema_version: Schema version to validate against for the specified collection.
            schema(optional): Schema to validate against (overrides collection schema).

        Returns:
            A tuple (bool, dict), where the boolean returns 'True' if the document
            is valid and 'False' otherwise. The dict returns HTTP response data, if
            the document fails JSON validation and '{}' otherwise.
        """
        is_valid = True
        details = {}
        error = None
        level = None

        # Save the document ID for reporting
        doc_id = None
        if '_id' in doc:
            doc_id = doc['_id']

        # Special case to prevent a document with no _id field from being written by a POST request
        if doc_id is None and "-CFG" in collection_name and "PONMGR-CFG" != collection_name and schema is None:
            is_valid = False
            details = {
                "level": "error",
                "message": "JSON validation failed: '_id' is a required property",
                "collection": collection_name,
                "id": doc_id,
                "path": "['_id']",
                "bad value": None,
                "bad value type": None,
                "validator": "required",
                "schema": self._get_schema(schema_version=schema_version, collection_name=collection_name)['properties']['_id']
            }
        else:
            # Validate the document against the collection's schema
            errors = None
            validator = self._get_validator(schema_version, collection_name, schema)
            if validator:
                errors = validator.iter_errors(doc)

            # Check for schema validation failures
            if errors:
                # ERROR - JSON Schema Validation Failure
                for err in errors:
                    # Check for "required" validation errors and check if they are enabled.
                    # If required validation is enabled, mark the error as a warning, but
                    # continue looking for errors. We want to return the first warning
                    # encountered.
                    # Note that required warnings are not considered 'errors'
                    # (e.g., 400 BAD REQUEST).
                    if err.validator == "required" and not strict_validation:
                        if validate_required and not error:
                            error = err
                            level = "warning"
                        else:
                            continue
                    else:
                        # Other validation failures are considered errors, break out
                        # now and return the error.
                        is_valid = False
                        error = err
                        level = "error"
                        break

            # Build a response payload for the POST/PUT HTTP response.
            if error:
                path = ""
                for elem in error.path:
                    path += f"['{elem}']"
                details = {
                    "level": level,
                    "message": f"JSON validation failed: {error.message}",
                    "collection": collection_name,
                    "id": doc_id,
                    "path": path,
                    "bad value": error.instance,
                    "bad value type": self._get_jsonschema_type(error.instance),
                    "validator": error.validator,
                    "schema": error.schema
                }

        return is_valid, details


    def validate_many(self, doc_dir: str=None, collection_name: str=None, doc_id: str=None, validate_required: bool=True, schema_version: str="current", verbose: bool=False):
        """
        Validate a set of documents in the specified directory against the JSON schema defined
        for each document's MongoDB collection.

        Args:
            doc_dir: The directory containing the documents to perform JSON schema validation on.
            collection_name: The collection name filter: ALL, CFG, STATE, or a
                specific collection (e.g., ONU-CFG).
            doc_id: The MongoDB document _id to filter on.
            validate_required: Check for missing 'required' fields.
            verbose: Enable verbose error reporting.

        Returns:
            A tuple (bool, List[dict]), where the boolean returns 'True' if all documents
            are valid and 'False' otherwise. The List[dict] returns details data for each
            document fails JSON validation and '[]' otherwise.
        """
        is_valid = False
        details = []
        success_count = 0
        failure_count = 0

        # Get the list of collections based on the specified collection name filter
        cnames = self._filter_collections(schema_version, collection_name)

        # Validate each document in the specified directory
        for file_name in sorted(os.listdir(doc_dir)):
            # Filter collection: if a collection was specified, skip documents that don't
            # match the specified collection
            file_path = f"{doc_dir}/{file_name}"
            try:
                file_collection = list(filter(file_name.startswith, self.collections[schema_version]))[0]
            except IndexError:
                file_collection = None

            # Filter Collection Name: if a document's collection does not match the filter, skip it.
            if file_collection not in cnames:
                continue

            # Load the document from the file
            if os.path.isfile(file_path) and file_name.endswith(".json"):
                with open(file_path, 'r', encoding="utf-8") as file:
                    doc = json.load(file)

            # Filter ID: if a document ID was specified, skip documents that don't
            # match the specified ID
            if doc_id and doc['_id'] != doc_id:
                continue

            # Validate the document
            doc_is_valid, doc_details = self.validate(file_collection, doc, validate_required, schema_version)
            if doc_is_valid:
                print(f"SUCCESS: {file_collection} document \'{file_name}\' is valid.")
                success_count += 1
            else:
                print(f"ERROR:   {file_collection} document \'{file_name}\' validation failed, {doc_details['message']}.")
                failure_count += 1
                if verbose:
                    err_msg = self.json_dumps(doc_details, sort_keys=False)
                    err_msg = err_msg.replace("\n", "\nERROR: ")
                    print(f"ERROR: details = {err_msg}")
                is_valid = False
                details.append(doc_details)

        print(f"Total success {success_count}, failure {failure_count}")

        return is_valid, details

    def validate_mongodb(self, db_uri:str, collection_name: str, doc_id: str = None, export_path: str = None, validate_required: bool = True, schema_version: str="current", verbose: bool = False):
        """
        Validate a document against the JSON schema defined for the specified
        MongoDB collection.

        Args:
            db_uri: MongoDB server URI and database name to connect to.
            collection_name: The collection name filter: ALL, CFG, STATE, or a
                specific collection (e.g., ONU-CFG).
            doc_id: The MongoDB document _id to filter on.
            export_path: Optional directory path to export documents from MongoDB.
            validate_required: Check for missing 'required' fields.
            verbose: Enable verbose error reporting.

        Returns:
            A tuple (bool, List[dict]), where the boolean returns 'True' if all documents
            are valid and 'False' otherwise. The List[dict] returns details data for each
            document fails JSON validation and '[]' otherwise.
        """
        is_valid = True
        details = []
        success_count = 0
        failure_count = 0

        # Get the list of collections based on the specified collection name filter
        cnames = self._filter_collections(schema_version, collection_name)

        # Connect to MongoDB
        # Note: only import pymongo if we're going to use it.
        import pymongo
        db = pymongo.MongoClient(db_uri).get_default_database()
        db_query = {}
        if doc_id:
            db_query["_id"] = doc_id

        # Optional export - create the export directory if it does not already exist
        if export_path:
            pathlib.Path(export_path).mkdir(parents=True, exist_ok=True)

        # For each collection, validate each document in MongoDB
        for cname in cnames:
            for doc in db[cname].find(db_query):
                # Validate the document from MongoDB
                doc_is_valid, doc_details = self.validate(
                    collection_name=cname,
                    doc=doc,
                    validate_required=validate_required,
                    schema_version=schema_version)
                if doc_is_valid:
                    print(f"SUCCESS: {cname} document \'{doc['_id']}\' is valid.")
                    success_count += 1
                else:
                    print(f"ERROR:   {cname} document \'{doc['_id']}\' validation failed, {doc_details['message']}.")
                    failure_count += 1
                    if verbose:
                        err_msg = self.json_dumps(doc_details, sort_keys=False)
                        err_msg = err_msg.replace("\n", "\nERROR: ")
                        print(f"ERROR: details = {err_msg}")
                    is_valid = False
                    details.append(doc_details)

                # Optional export - write out the document to the file system.
                if export_path and doc['_id'] is not None:
                    exported_doc_id = doc['_id']
                    exported_doc_id.replace(":","")
                    exported_file_path = f"{export_path}/{cname}-{exported_doc_id}.json"
                    with open(exported_file_path, 'w', encoding="utf-8") as fp:
                        json.dump(doc, fp)

        print(f"Total success {success_count}, failure {failure_count}")

        return is_valid, details

    def validate_path(self, collection_name: str, path: str, value: str=None, schema_version: str="current", schema: dict=None) -> Tuple[bool, dict]:
        """
        Validate a MongoDB query path using dot notation against the JSON schema
        defined for the specified MongoDB collection. This method also validates
        an optional value against the schema.

        Example: only the path needs to be validated given the following API query
        parameter passed to the endpoint:
            "projection=OLT.MAC Address"

        Example: bot the path and value need to be validated given the following API
        query parameter passed to the endpoint:
            "query=OLT.MAC Address=e8:b4:70:70:0c:9c"

        Args:
            collection_name: The name of the collection in MongoDB (e.g., ONU-CFG).
            path: The MongoDB query path in dotted notation (e.g., ONU.Name) to validate.
            value(optional): The query match value to validate against the schema.
            schema_version: Schema version to validate against for the specified collection.
            schema(optional): Schema to validate against (overrides collection schema).

        Returns:
            A tuple (bool, dict), where the boolean returns 'True' if the document
            is valid and 'False' otherwise. The dict returns HTTP response data, if
            the document fails JSON validation and '{}' otherwise.
        """
        is_valid = False
        details = {}

        try:
            # Lookup the schema for this collection if not specified
            if not schema:
                schema = self._get_schema(schema_version, collection_name)

            # Lookup the node for the specified path
            sub_schema = None
            if schema:
                sub_schema = self._get_node(schema, path)

            # Attempt to validate the value against the sub-schema
            if sub_schema:
                if value:
                    validator = self._get_validator(schema_version, collection_name, sub_schema)
                    if validator:
                        validator.validate(value)
                is_valid = True
        except (jsonschema.SchemaError, jsonschema.ValidationError) as err:
            # ERROR - JSON schema validation failed for the query value
            # Build a response payload for the POST/PUT HTTP response.
            err_path = ""
            for elem in err.path:
                err_path += f"['{elem}']"
            details = {
                "level": "error",
                "message": f"JSON validation failed: {err.message}",
                "collection": collection_name,
                "path": err_path,
                "bad value": err.instance,
                "bad value type": self._get_jsonschema_type(err.instance),
                "validator": err.validator,
                "schema": err.schema
            }

        except jsonschema.RefResolutionError as err:
            print(f"ERROR: JSON schema reference resolution failure.")
            print(f"ERROR: {err}")
            details = {
                "level": "error",
                "message": f"JSON schema reference resolution failure for {collection_name}, path {path}.",
                "cause": f"{err}",
                "collection": collection_name,
                "path": path
            }

        except KeyError:
            # ERROR - Invalid query path
            # Build a response payload for the POST/PUT HTTP response.
            details = {
                "level": "error",
                "message": f"Invalid query path for {collection_name}, path {path}.",
                "collection": collection_name,
                "path": path
            }

        return is_valid, details

    def extract_schema(self, db_uri:str, collection_name: str, doc_id: str=None):
        """
        Validate a document against the JSON schema defined for the specified
        MongoDB collection.

        Args:
            db_uri: MongoDB server URI and database name to connect to.
            collection_name: The name of the collection in MongoDB (e.g., ONU-CFG).
            doc_id: The MongoDB document _id to filter on.

        Returns:
            A string contained the exported schema.
        """
        # Only import pymongo and genson if we're going to use it.
        import pymongo
        import genson
        db = pymongo.MongoClient(db_uri).get_default_database()
        db_query = {}
        if doc_id:
            db_query["_id"] = doc_id

        schema_builder = genson.SchemaBuilder()
        if collection_name in self.schemas['current']:
            schema_builder.add_schema(self.schemas['current'][collection_name])
        for doc in db[collection_name].find(db_query):
            schema_builder.add_object(doc)

        return schema_builder.to_schema()

    def flatten(self, data: dict, sep: str=".") -> dict:
        """
        Flatten a JSON document into a dictionary containing key = value pairs.
        The key is a JSON path for a field represented in dotted notation.

        Args:
            data: The JSON data to flatten.
            sep: The separator character to use in the flattened key.

        Returns:
            A flattened, key = value representation of the data.
        """
        flat_data = {}

        def _flatten(node, name=''):
            if isinstance(node, dict):
                for key in node:
                    _flatten(node[key], name + key + sep)
            elif isinstance(node, list):
                index = 0
                for elem in node:
                    _flatten(elem, name + str(index) + sep)
                    index += 1
            else:
                flat_data[name[:-1]] = node

        _flatten(data)
        return flat_data

    def json_dumps(self, data, sort_keys=True, indent=4, separators=(',', ': ')) -> str:
        """ Pretty print JSON data/dict to a string. """
        return dumps(data, sort_keys=sort_keys, indent=indent, separators=separators)

    def schema_exists(self, schema_version: str, collection_name: str)  -> bool:
        """ Return true if a schema exists for the specified collection. """
        exists = True
        cnames = self._filter_collections(schema_version, collection_name)
        if cnames:
            for cname in cnames:
                if not self._get_schema(schema_version, cname):
                    exists = False
                    break
        else:
            exists = False

        return exists

    def get_schema_version_for_doc_version(self, doc_version: str) -> str:
        """ Look up the schema version from a document version. """
        schema_version = None
        if doc_version:
            # Strip the release type letter from the front.
            doc_version = doc_version[1:]
            # Split the parts <maj>.<min>.<patch>, since only the <maj>.<min> will be used for comparison
            items = doc_version.split('.')
            if len(items) >= 3:
                doc_version = ".".join(items[:2])
                schema_version = next((s for s in self.schema_versions if s[1:].startswith(doc_version)), None)
        return schema_version

    def _is_schema_version_dir(self, path_name: str) -> bool:
        """ Return true if path name has the format 'R<major>.<minor>.<patch>'. """
        try:
            valid = bool(re.match(r'^[ABER][0-9]*\.[0-9]*\.[0-9]*$', path_name))
        except TypeError:
            valid = False
        return valid

    def _get_jsonschema_type(self, val: Any) -> str:
        """ Get the JSON Schema type for the specified value. """
        jsonschema_type = "unknown"
        if isinstance(val, list):
            jsonschema_type = "array"
        elif isinstance(val, bool):
            jsonschema_type = "boolean"
        elif isinstance(val, int):
            jsonschema_type = "integer"
        elif isinstance(val, dict):
            jsonschema_type = "object"
        elif isinstance(val, float):
            jsonschema_type = "number"
        elif isinstance(val, str):
            jsonschema_type = "string"
        elif val is None:
            jsonschema_type = "null"
        return jsonschema_type

    def _get_ref(self, schema: dict, path: str) -> dict:
        """ Resolve a reference for path validation. """
        node = schema
        for segment in path.split('/')[1:]:
            node = node[segment]
        return node

    def _get_node(self, schema: dict, path: str) -> dict:
        """ Return the schema node for the specified path. """
        node = schema
        for segment in path.split('.'):
            # Security check
            # '$' Do not allow paths that contain '$', except for
            # projection positional '$' by itself.
            if '$' in segment and segment != '$':
                raise KeyError

            # Resolve references if needed
            if '$ref' in node:
                node = self._get_ref(schema, node['$ref'])

            # Get the node for this segment of the path
            if 'properties' in node and segment in node['properties']:
                # Look up node for an exact key in a nested dictionary.
                node = node['properties'][segment]
            elif 'patternProperties' in node:
                # Look up node for a regex key in a nested dictionary.
                is_match = False
                for key in node['patternProperties'].keys():
                    is_match = bool(re.match(key, segment))
                    # print(f"  patternProperties regex {key}, is_match {is_match}")
                    if is_match:
                        node = node['patternProperties'][key]
                        break
                # If there is no match, consider this a failure
                if not is_match:
                    raise KeyError
            elif (segment == '$' or self._is_int(segment)) and 'items' in node:
                # Look up node for an index in an array
                node = node['items']
            elif 'items' in node and 'properties' in node['items']:
                # Look up node for an array of objects
                node = node['items']['properties'][segment]
            else:
                raise KeyError

        return node

    def _get_node_for_key_path(self, schema: dict, path: str) -> dict:
        """ Return the schema node for the specified path. """
        try:
            return self._get_node(schema, path)
        except KeyError:
            return None

    def _is_int(self, val: Any) -> bool:
        """ Return true if the value resolves to a Python integer. """
        try:
            int(val)
            rc = True
        except ValueError:
            rc = False
        return rc

    def _is_float(self, val: Any) -> bool:
        """ Return true if the value resolves to a Python floating point number. """
        try:
            float(val)
            rc = True
        except ValueError:
            rc = False
        return rc

    def _parse_value(self, json_val: Any) -> Any:
        """ Parse and convert a JSON value into a Python data type. """
        if json_val == 'true':
            val = True
        elif json_val == 'false':
            val = False
        elif self._is_int(json_val):
            val = int(json_val)
        elif self._is_float(json_val):
            val = float(json_val)
        else:
            val = json_val
        return val

    def _get_schema(self, schema_version: str, collection_name: str) -> Optional[dict]:
        """ Get a schema for a specific version and collection. """
        schema = None
        if schema_version and schema_version in self.schemas:
            if collection_name and collection_name in self.schemas[schema_version]:
                schema = self.schemas[schema_version][collection_name]
        return schema

    def _get_validator(self, schema_version: str, collection_name: str, sub_schema: dict=None) -> Optional[object]:
        """ Get a schema validator for a specific version and collection. """
        validator = None
        resolver = None

        try:
            # Create a custom validator with the following overrides:
            #  - override the jsonschema 'integer' type checker to require an python integer data type
            #  - override the 'required' validator to allow for skipping required checks (e.g., PATCH)
            type_checker = jsonschema.Draft7Validator.TYPE_CHECKER.redefine("integer", is_int)
            CustomDraft7Validator = jsonschema.validators.extend(
                jsonschema.Draft7Validator,
                validators={"required": required_validator},
                type_checker=type_checker)

            # Create a schema store constisting of COMMON-TYPES, which contains
            # type definitions shared accross all schema files.
            schema_store = {}
            if 'COMMON-TYPES' in self.schemas[schema_version]:
                if '$id' in self.schemas[schema_version]['COMMON-TYPES']:
                    schema_store[self.schemas[schema_version]['COMMON-TYPES']['$id']] = self.schemas[schema_version]['COMMON-TYPES']

            # Set up a validator if the schema is available for this collection
            schema = self._get_schema(schema_version, collection_name)
            if schema:
                resolver = jsonschema.RefResolver.from_schema(schema, store=schema_store)
                validator = CustomDraft7Validator(
                    schema,
                    resolver=resolver,
                    format_checker=jsonschema.draft7_format_checker)

            # If a sub_schema was specified, set up a validator from the sub_schema
            if sub_schema:
                # If there's no collection validator, set up a standard Draft7 Validator
                if not validator:
                    resolver = jsonschema.RefResolver.from_schema(sub_schema, store=schema_store)
                    validator = CustomDraft7Validator

                # Extend the collection validator or the CustomDraft7Validator
                CustomValidator = jsonschema.validators.extend(validator)

                # Set up a sub_validator for this sub_schema (and return it as the validator for this call)
                validator = CustomValidator(
                    sub_schema,
                    resolver=resolver,
                    format_checker=jsonschema.draft7_format_checker)

        except jsonschema.SchemaError as err:
            print(f"ERROR: JSON schema is invalid.")
            print(f"ERROR: {err}")

        return validator

    def _filter_collections(self, schema_version: str, collection_name):
        """  Get the list of collections based on the specified collection name filter. """
        if collection_name.upper() == 'ALL':
            cnames = self.collections[schema_version]
        elif collection_name.upper() == 'CFG':
            cnames = list(filter(lambda x:x.endswith("CFG"), self.collections[schema_version]))
        elif collection_name.upper() == 'STATE':
            cnames = list(filter(lambda x:x.endswith("STATE"), self.collections[schema_version]))
        else:
            cnames = [collection_name]
        return cnames


def _main():
    import argparse
    import sys

    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-c", "--collection", action="store", dest="collection", default=None, required=True, help="MongoDB Collection to validate against.")
    parser.add_argument(      "--db", action="store", dest="db_uri", default=None, required=False, help="Path to the JSON document to validate.")
    parser.add_argument("-d", "--document", action="store", dest="doc", default=None, required=False, help="Path to the JSON document to validate.")
    parser.add_argument(      "--export-path", action="store", dest="export_path", default=False, required=False, help="Export documents from MongoDB during validation.")
    parser.add_argument(      "--extract-schema", action="store_true", dest="extract_schema", default=False, required=False, help="Reverse engineer a JSON schmea from MongoDB.")
    parser.add_argument(      "--flatten", action="store_true", dest="flatten", default=False, required=False, help="Reverse engineer a JSON schmea from MongoDB.")
    parser.add_argument(      "--id", action="store", dest="doc_id", default=None, required=False, help="Document ID to validate in MongoDB.")
    parser.add_argument(      "--query-path", action="store", dest="query_path", default=None, required=False, help="Path to a JSON schema node to validate")
    parser.add_argument(      "--query-value", action="store", dest="query_value", default=None, required=False, help="Value to validate (works in conjunction with --query-path)")
    parser.add_argument("-s", "--schema-path", action="store", dest="schema_path", default=None, required=False, help="Path to JSON schemas used for validation. If not specified validation is a simple JSON lint/format check.")
    parser.add_argument(      "--skip-required", action="store_true", dest="skip_required", default=False, required=False, help="Skip validation for 'required' fields.")
    parser.add_argument(      "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.add_argument("-v", "--version", action="store", dest="schema_version", default="current", required=False, help="Schema version (e.g., R2.3.0)")
    parser.parse_args()
    args = parser.parse_args()

    # Load the JSON document
    doc = None
    if args.doc and os.path.isfile(args.doc):
        with open(args.doc, 'r', encoding="utf-8") as f:
            try:
                doc = json.load(f)
            except json.JSONDecodeError as err:
                print(f"ERROR: JSON document format/lint failed for {args.doc}.")
                print(f"ERROR: {err}")
                sys.exit(1)

    validator = None
    if args.schema_path:
        validator = JsonSchemaValidator(args.schema_path)

    # Generate a warning if there are no validators for the specified collection
    if not validator.schema_exists(args.schema_version, args.collection):
        print(f"WARNING: No schema validator for {args.schema_version}, {args.collection}!")

    # Validate a single document
    if validator and doc:
        valid, details = validator.validate(
            collection_name=args.collection,
            doc=doc,
            validate_required=not args.skip_required,
            schema_version=args.schema_version)
        if not valid:
            print(f"ERROR: JSON validation failed for {details['collection']}, doc_id {details['id']}.")
            print(f"ERROR: {details['message']}")
            err_msg = validator.json_dumps(details, sort_keys=False)
            err_msg = err_msg.replace("\n", "\nERROR: ")
            print(f"ERROR: details = {err_msg}")
            print("Failed.")
            sys.exit(1)
        elif details:
            print(f"WARNING: JSON validation warning for {details['collection']}, doc_id {details['id']}.")
            print(f"WARNING: {details['message']}")
            err_msg = validator.json_dumps(details, sort_keys=False)
            err_msg = err_msg.replace("\n", "\nWARNING: ")
            print(f"WARNING: details = {err_msg}")
            # Warnings are considered success

    # Flatten a single document
    if validator and args.flatten and doc:
        data = validator.flatten(doc)
        print(validator.json_dumps(data, sort_keys=False))
        # Success
        print("Success.")
        sys.exit(0)

    # Validate query parameters
    if validator and args.query_path:
        valid, details = validator.validate_path(
            collection_name=args.collection,
            path=args.query_path,
            value=args.query_value,
            schema_version=args.schema_version)
        if not valid and 'bad value' in details:
            print(f"ERROR: Invalid query value for {details['collection']}, path {details['path']}, value {details['bad value']}.")
            print(f"ERROR: {details['message']}")
            err_msg = validator.json_dumps(details, sort_keys=False)
            err_msg = err_msg.replace("\n", "\nERROR: ")
            print(f"ERROR: details = {err_msg}")
            print("Failed.")
            sys.exit(1)
        elif not valid:
            print(f"ERROR: Invalid query path for {details['collection']}, path {details['path']}.")
            # print(f"ERROR: {details['message']}")
            err_msg = validator.json_dumps(details, sort_keys=False)
            err_msg = err_msg.replace("\n", "\nERROR: ")
            print(f"ERROR: details = {err_msg}")
            print("Failed.")
            sys.exit(1)

    # Extract JSON Schema from documents in MongoDB
    if validator and args.extract_schema:
        extracted_schema = validator.extract_schema(
                db_uri=args.db_uri,
                collection_name=args.collection,
                doc_id=args.doc_id)
        print(f"{args.collection} Schema:")
        print(validator.json_dumps(extracted_schema, sort_keys=False))
        # Success
        print("Success.")
        sys.exit(0)

    # Validate documents from MongoDB
    if validator and args.db_uri:
        valid, details = validator.validate_mongodb(
                db_uri=args.db_uri,
                collection_name=args.collection,
                doc_id=args.doc_id,
                export_path=args.export_path,
                validate_required=False,
                schema_version=args.schema_version,
                verbose=args.verbose)
        if not valid:
            print("Failed.")
            sys.exit(1)

    # Validate documents from a directory
    if args.doc and os.path.isdir(args.doc):
        valid, details = validator.validate_many(
            args.doc,
            args.collection,
            doc_id=args.doc_id,
            validate_required=False,
            schema_version=args.schema_version,
            verbose=args.verbose)
        if not valid:
            print("Failed.")
            sys.exit(1)

    if not validator:
        print("Skipping JSON schema validation.")

    print("Success.")
    sys.exit(0)


# Debug
if __name__ == '__main__':
    _main()
