"""
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import collections
from pymongo import monitoring


class CommandStatistics:
    """ A Command listener command """
    def __init__(self, command_name):
        self.command_name = command_name
        self.average_latency = 0
        self.min_latency = 0
        self.max_latency = 0
        self.success_count = 0
        self.failure_count = 0
        self.latency_sample_set = collections.deque(250 * [0], 250)
    def _incr64(self, count):
        """ Check if counter has reached max signed 64-bit integer value. If so, rollover to 1 """
        if not abs(count) > (1 << 63) - 2:  # sub 2 so we rotate 1 number before exceeding max signed 64 bit int
            count += 1
        else:
            count = 1
        return count

    def _record_latency(self, latency):
        """ Record the latency of a given DB command """
        latency = latency / 1000  # Convert to microseconds from milliseconds
        self.latency_sample_set.pop()
        self.latency_sample_set.appendleft(round(latency, 0))

    def _calculate_latency(self):
        """ Calculate the latency of a given DB command """
        self.max_latency = max(self.latency_sample_set)
        non_zero_sample_set_count = len(self.latency_sample_set) - self.latency_sample_set.count(0)
        if non_zero_sample_set_count > 0:
            self.min_latency = min(latency for latency in self.latency_sample_set if latency != 0)
            self.average_latency = round((sum(self.latency_sample_set) / non_zero_sample_set_count), 0)

    def success(self, duration_micros):
        """ Handler for successful DB commands """
        self.success_count = self._incr64(self.success_count)
        self._record_latency(duration_micros)

    def failure(self, duration_micros):
        """ Handler for failed DB commands """
        self.failure_count = self._incr64(self.failure_count)
        self._record_latency(duration_micros)

    def get_statistics(self):
        """ Gets command statistics for the given command
        Returns:
            dict: All command statistics in DB format
        """
        self._calculate_latency()
        return {
            'Average Latency': self.average_latency,
            'Min Latency': self.min_latency,
            'Max Latency': self.max_latency,
            'Success Count': self.success_count,
            'Failure Count': self.failure_count
        }

class CustomCommandLogger(monitoring.CommandListener):
    """A simple listener that logs command events.
    Listens for :class:`~pymongo.monitoring.CommandStartedEvent`,
    :class:`~pymongo.monitoring.CommandSucceededEvent` and
    :class:`~pymongo.monitoring.CommandFailedEvent` events and
    logs them at the `INFO` severity level using :mod:`logging`.
    """

    def __init__(self):
        self.commands_statistics = {} # Dictionary containing all command statistics

    def succeeded(self, event: monitoring.CommandSucceededEvent) -> None:
        """Listener for succeeded DB Commands. """
        if event.command_name not in self.commands_statistics:
            self.commands_statistics[event.command_name] = CommandStatistics(event.command_name)
        self.commands_statistics[event.command_name].success(event.duration_micros)

    def failed(self, event: monitoring.CommandFailedEvent) -> None:
        """Listener for failed DB Commands. """
        if event.command_name not in self.commands_statistics:
            self.commands_statistics[event.command_name] = CommandStatistics(event.command_name)
        self.commands_statistics[event.command_name].failure(event.duration_micros)

    def started(self, event: monitoring.CommandStartedEvent):
        """ Not currently used. """
        pass
    def get_statistics(self):
        """ Gets all command statistics
        Returns:
            dict: All command statistics in DB format
        """
        stats = {}
        for name, command in self.commands_statistics.items():
            stats[name.capitalize()] = command.get_statistics()
        self.clear_statistics()
        return stats

    def clear_statistics(self):
        self.commands_statistics = {}
