#!/usr/bin/env python3
#--------------------------------------------------------------------------#
#  Copyright (c) 2020-2024, Ciena Corporation                              #
#  All rights reserved.                                                    #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
#  PROPRIETARY NOTICE                                                      #
#  This Software consists of confidential information.                     #
#  Trade secret law and copyright law protect this Software.               #
#  The above notice of copyright on this Software does not indicate        #
#  any actual or intended publication of such Software.                    #
#                                                                          #
#--------------------------------------------------------------------------#

""" Get the registration status for ONUs.

This Tibit YANG Example script retrieves ONU registration status for an OLT.

Example - Get the ONU registration status for an OLT:

  ./get_onu_registration_status.py --olt 70:b3:d5:52:37:24

Example - Get ONUs with registration state 'disallowed-admin' for an OLT:

  ./get_onu_registration_status.py --olt 70:b3:d5:52:37:24 --filter disallowed-admin


usage: get_onu_registration_status.py [-f FILTER] [--help] [-h HOST] --olt OLT
                                      [-w PASSWD] [-p PORT] [-u USER] [-v]

optional arguments:
  -f FILTER, --filter FILTER
                        Filter by ONU state (e.g., deregistered, disabled,
                        disallowed-admin, disallowed-error, dying-gasp,
                        registered, unprovisioned, unspecified) (default:
                        None)
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --olt OLT             OLT MAC Address (e.g., 70:b3:d5:52:37:24) (default:
                        None)
  -w PASSWD, --passwd PASSWD
                        Password. If no password is provided, the user will be
                        prompted to enter. (default: None)
  -p PORT, --port PORT  NETCONF Server port number. (default: 830)
  -u USER, --user USER  Username. (default: None)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
import sys
from lxml import etree
from netconf_driver import NetconfDriver

if __name__ == '__main__':
    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("-f", "--filter", action="store", dest="filter", default=None, required=False, help="Filter by ONU state (e.g., deregistered, disabled, disallowed-admin, disallowed-error, dying-gasp, registered, unprovisioned, unspecified)")
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--olt", action="store", dest="olt", default=None, required=True, help="OLT MAC Address (e.g., 70:b3:d5:52:37:24)")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, the user will be prompted to enter.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Build an options dictionary from the command line arguments
    options = {
        "{{OLT}}" : args.olt
    }

    # Send a Netconf <get> request to retreive the ONU states for an OLT
    ONU_REG_STATUS = '''
    <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
        <get>
            <filter type="subtree">
                <tibitcntlr:pon-state xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db">
                    <tibitcntlr:olt-state>
                    <tibitcntlr:olt>
                        <name>{{OLT}}</name>
                        <tibitcntlr:onu-states/>
                    </tibitcntlr:olt>
                    </tibitcntlr:olt-state>
                </tibitcntlr:pon-state>
            </filter>
        </get>
    </rpc>
    '''
    rsp_xml = nc.get(data_xml=ONU_REG_STATUS, options=options, message="/tibit-pon-controller-db::pon-state/tibit-pon-controller-db:olt-state/tibit-pon-controller-db:olt/tibit-pon-controller-db:onu-states")

    # Parse the Netconf response and retrieve and build a dictionary of ONUs with
    # states from the XML response data.
    onu_registration_status = {}
    if rsp_xml:
        NSMAP = {
            'nc' : "urn:ietf:params:xml:ns:netconf:base:1.0",
            'tibit' : "urn:com:tibitcom:ns:yang:controller:db",
            }
        root = etree.fromstring(rsp_xml)
        # If a filter was specified, only display ONUs with a state that matches the filter
        if args.filter:
            onu_states = [ args.filter ]
        else:
            onu_states = [
                # There are situations where the ONU can appear in 'unprovisioned' or 'unspecified'
                # in addition to other states. As a result, processes these states first, then
                # overwrite later with the actual state.
                "unprovisioned",
                "unspecified",
                # ONUs appear in the remaining states only once.
                "deregistered",
                "disabled",
                "disallowed-admin",
                "disallowed-error",
                "dying-gasp",
                "registered",
            ]
        for onu_state in onu_states:
            for onu in root.findall(f"nc:data/tibit:pon-state/tibit:olt-state/tibit:olt/tibit:onu-states/tibit:{onu_state}/tibit:id", namespaces=NSMAP):
                onu_registration_status[onu.text] = onu_state

    # Display the ONU registration status of for the OLT
    print(f"\nONU Registration Status for OLT {args.olt}:")
    for onu,state in sorted(onu_registration_status.items()):
        print(f"{onu} = {state}")

