#!/usr/bin/env python3
#--------------------------------------------------------------------------#
#  Copyright (c) 2020-2024, Ciena Corporation                              #
#  All rights reserved.                                                    #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
#  PROPRIETARY NOTICE                                                      #
#  This Software consists of confidential information.                     #
#  Trade secret law and copyright law protect this Software.               #
#  The above notice of copyright on this Software does not indicate        #
#  any actual or intended publication of such Software.                    #
#                                                                          #
#--------------------------------------------------------------------------#

""" Disable Service for an ONU.

This Tibit YANG Example script disables service for an ONU. The script also
provides an option to delete the ONU configuration from the database.

Example:

    ./add_ctag_service/edit_config_disable_svc.py \
        --olt 70:b3:d5:52:37:24 \
        --onu ALPHe30cadcf \
        --olt_tag 200 \
        --onu_tag 25


usage: edit_config_disable_svc.py [--delete] [--help] [-h HOST] --olt OLT
                                  [--olt_tag OLT_TAG] --onu ONU --onu_tag
                                  ONU_TAG [--option_82 OPTION_82] [-w PASSWD]
                                  [-p PORT] [-u USER] [-v]

optional arguments:
  --delete              Deletes the ONU configuration from the database.
                        (default: False)
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --olt OLT             OLT MAC Address (e.g., 70:b3:d5:52:37:24) (default:
                        None)
  --olt_tag OLT_TAG     Tag to be added by the OLT (default: 0)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  --onu_tag ONU_TAG     Tag to be added by the ONU (default: None)
  --option_82 OPTION_82
                        Option 82 Value (hex string) (default: None)
  -w PASSWD, --passwd PASSWD
                        Password. If no password is provided, the user will be
                        prompted to enter. (default: None)
  -p PORT, --port PORT  NETCONF Server port number. (default: 830)
  -u USER, --user USER  Username. (default: None)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
import os
import sys
sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
from netconf_driver import NetconfDriver
from add_ctag_service.edit_config_add_ctag_svc import get_nni_network_id_from_tags

if __name__ == '__main__':
    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--delete", action="store_true", dest="delete", default=False, required=False, help="Deletes the ONU configuration from the database.")
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--olt", action="store", dest="olt", default=None, required=True, help="OLT MAC Address (e.g., 70:b3:d5:52:37:24)")
    parser.add_argument(      "--olt_tag", action="store", dest="olt_tag", default=None, required=True, help="Tag to be added by the OLT")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument(      "--onu_tag", action="store", dest="onu_tag", default=None, required=True, help="Tag to be added by the ONU")
    parser.add_argument(      "--option_82", action="store", dest="option_82", default=None, required=False, help="Option 82 Value (hex string)")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, the user will be prompted to enter.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Build an options dictionary from the command line arguments
    options = {
        "{{OLT}}" : args.olt,
        "{{OLT_TAG}}" : args.olt_tag,
        "{{ONU}}" : args.onu,
        "{{ONU_TAG}}" : args.onu_tag,
    }

    # Disable the service or delete the ONU configuration depending on the command line options
    if args.delete:
        # Delete the ONU-CFG from the database
        nc.edit_config(
            message = "onu-cfg",
            data_xml = '''
            <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
            <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
                <target>
                    <running/>
                </target>
                <config>
                    <!-- Delete the ONU Configuration from the database -->
                    <pon xmlns="urn:com:tibitcom:ns:yang:controller:db">
                        <onus>
                            <onu xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0" nc:operation="remove">
                                <name>{{ONU}}</name>
                                <device-id>{{ONU}}</device-id>
                            </onu>
                        </onus>
                    </pon>
                </config>
            </edit-config>
            </rpc>
            ''',
            options=options
        )
    else:
        # Disable the service in ONU-CFG
        nc.edit_config(
            message = "onu-cfg",
            data_xml = '''
            <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
            <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
                <target>
                    <running/>
                </target>
                <config>
                    <!-- Disable the service, OLT Service Port and Networks -->
                    <pon xmlns="urn:com:tibitcom:ns:yang:controller:db">
                        <onus>
                            <onu>
                                <name>{{ONU}}</name>
                                <device-id>{{ONU}}</device-id>
                                <olt xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0" nc:operation="replace">
                                </olt>
                                <olt-service xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0" nc:operation="replace">
                                    <id>0</id>
                                    <dhcp-relay>
                                    <circuit-id/>
                                    <remote-id/>
                                    <sub-options/>
                                    </dhcp-relay>
                                    <enable>false</enable>
                                    <name/>
                                    <protocol-filters>
                                        <dhcpv4>pass</dhcpv4>
                                        <dhcpv6>pass</dhcpv6>
                                        <eapol>pass</eapol>
                                    </protocol-filters>
                                    <sla-profile>Max</sla-profile>
                                </olt-service>
                                <olt-service>
                                    <id>1</id>
                                    <enable>false</enable>
                                </olt-service>
                                <olt-service>
                                    <id>2</id>
                                    <enable>false</enable>
                                </olt-service>
                                <olt-service>
                                    <id>3</id>
                                    <enable>false</enable>
                                </olt-service>
                                <olt-service>
                                    <id>4</id>
                                    <enable>false</enable>
                                </olt-service>
                                <olt-service>
                                    <id>5</id>
                                    <enable>false</enable>
                                </olt-service>
                                <olt-service>
                                    <id>6</id>
                                    <enable>false</enable>
                                </olt-service>
                                <olt-service>
                                    <id>7</id>
                                    <enable>false</enable>
                                </olt-service>
                                <onu>
                                    <cvlan-id>1</cvlan-id>
                                    <service-config>DISABLED</service-config>
                                </onu>
                            </onu>
                        </onus>
                    </pon>
                </config>
            </edit-config>
            </rpc>
            ''',
            options=options
        )

    # Remove the ONU from inventory
    nc.edit_config(
        message = "olt-cfg-onu-inventory",
        data_xml = '''
        <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
        <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
            <target>
                <running/>
            </target>
            <config>
                <pon xmlns="urn:com:tibitcom:ns:yang:controller:db">
                    <olts xmlns="urn:com:tibitcom:ns:yang:controller:db">
                        <olt>
                            <name>{{OLT}}</name>
                            <onus xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0" nc:operation="remove">
                                <id>{{ONU}}</id>
                            </onus>
                        </olt>
                    </olts>
                </pon>
            </config>
        </edit-config>
        </rpc>
        ''',
        options=options
    )

    # Remove the NNI Network from inventory
    # Need to lookup the NNI Inventory entry id (index) from the VLAN Tags
    nni_network_id = get_nni_network_id_from_tags(nc, f"s{args.olt_tag}.c{args.onu_tag}.c0", options)
    if nni_network_id is not None:
        options['{{NNI_NETWORK_ID}}'] = nni_network_id
        nc.edit_config(
            message = "olt-cfg-nni-inventory",
            data_xml = '''
            <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760">
            <edit-config xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
                <target>
                    <running/>
                </target>
                <config>
                    <pon xmlns="urn:com:tibitcom:ns:yang:controller:db">
                        <olts>
                            <olt>
                                <name>{{OLT}}</name>
                                <nni-networks xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0" nc:operation="remove">
                                    <id>{{NNI_NETWORK_ID}}</id>
                                </nni-networks>
                            </olt>
                        </olts>
                    </pon>
                </config>
            </edit-config>
            </rpc>
            ''',
            options=options
        )

