#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2015 - 2021 by Tibit Communications, Inc.                  #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#

import os
import os.path
import sys
import argparse

this_dir = os.path.dirname(os.path.realpath(__file__)) # path to this file
sys.path.append(os.path.join(this_dir, ".."))

from tapi.tapi3.TibitApi import TibitApi, Options

if __name__ == '__main__':

    # Example command line invocation to discover Tibit OLTs using linux interface 'ens9f0.4090'
    #
    # sudo ./DiscoverOlts.py -i ens9f0.4090
    #
    # Adding the '-v' switch will generate debug output showing the details of the OAM message exchanges between TAPI and
    # the managed devices.
    parser = argparse.ArgumentParser(add_help=True, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument('-i', '--if', action='store', dest='interface', default="ens9f0", required=False, help="Name of local interface (e.g., ens9f0) connected to the OLT")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, help="If supplied verbose mode will be set to true.")
    parser.parse_args()
    args = parser.parse_args()

    #########################################
    # Instantiate Tibit Adapter
    #########################################

    # Instantiate an Options object to change the interface to match the interface connected to the OLT's NNI port
    options = Options(interface=args.interface, verbose=args.verbose)

    # Instantiate the Tibit Adapter using the Options object created above
    tapi = TibitApi(options)

    #########################################
    # Instantiate OLT Object
    #########################################

    # Discover the OLTs and provision them in TAPI
    olts = tapi.discoverAndProvisionOlts(interface=args.interface, timeout=3)

    if olts == []:
        print("No OLTs discovered on interface {}".format(args.interface))
    else:

        oltMacAddrs = tapi.getOltMacAddrs()
        print("Learned OLT MAC Addresses are {}".format(oltMacAddrs))

        for olt in olts:

            #############################################
            # Get List of ONUs registered on the OLT
            #############################################

            # Populate TAPI's knowledge of the ONUs and Links connected to the OLT
            onu_list = olt.getRegisteredOnus()


            #############################################
            # Query OLT Information
            #############################################

            # Display OLT information to the console
            olt.dump()
            print("LldpCaptureState {}".format(olt.nniPort.getLldpCaptureState()))
            print("MgmtTpid 0x{:x}".format(olt.getMgmtTpid()))
            print("MgmtVlan {}".format(olt.getMgmtVlan()))


            # Retrieve a dictionary with the OLT's attributes
            oltInfo = olt.getInfo()

            #############################################
            # Query ONU Information
            #############################################

            # Loop through all of the ONUs that were discovered
            if onu_list is not None:
                for onu in onu_list:

                    # Display ONU information to the console
                    onu.dump()

                    # Retrieve a dictionary with the ONU's attributes
                    onuInfo = onu.getInfo()

        # Cleanup OLTs
        del olts[:]
