#!/usr/bin/env bash
#--------------------------------------------------------------------------#
# Copyright (C) 2015 - 2022 by Tibit Communications, Inc.                  #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

VERSION=R5.2.0
INITDIR=$(pwd)
PONCNTLVER=R5.2.0
NETCONFVER=R5.2.0
PONMGRVER=R5.2.0
SNMPVERSION="1.0.10"
ControllerImage=tibit-poncntl_${PONCNTLVER}_amd64.deb
ManagerImage=tibit-ponmgr_${PONMGRVER}_all.deb
NetconfImage=tibit-netconf-confd_${NETCONFVER}_amd64.deb
SnmpImage=snmptibit-${SNMPVERSION}.deb
DATABASE_PORT=27017
PLATFORM=UB2004
LogFile=install.log

exec > >( tee -a "${LogFile}" )
exec 2> >( tee -a "${LogFile}" >&2 )

echo "Checking system compatibility....."
build_os="$(lsb_release -i -s)$(lsb_release -r -s)"
if [ "${build_os}" == "Ubuntu20.04" ]; then
  echo "Compatibility check succeeded....."
elif [ "${build_os}" == "Ubuntu18.04" ]; then
  echo "This system is running \"Ubuntu18.04\" and this installation is for \"Ubuntu20.04\"."
  echo "Please download the \"Ubuntu18.04\" installation bundle: ${VERSION}-UB1804.zip"
  exit 1
else
  echo "Ubuntu version \"${build_os}\" is not supported....."
  echo "Terminating installation ....."
  exit 1
fi

if [ "${build_os}" == "Ubuntu20.04" ]; then
  PLATFORM=UB2004
fi
if [ "${build_os}" == "Ubuntu22.04" ]; then
  PLATFORM=UB2204
fi

helpFunction()
# Print command usage and help information and exits
{
   echo 
   echo "Usage: $0 -e <l2EthInterfaceName> 

Installs (PON Manager + MongoDB + NetConf + SNMP) and/or PON Controller

   required: 
       -e <l2EthInterfaceName>   interface name of L2 port

   optional: 
       -d <databaseIpAaddress>   IP address of MongoDB database
       -n <databaseName>         MongoDB database name for PON Manager
       -m                        Install PonManager/MongoDB/NETCONF only
       -c                        Install PonController only
       -s                        To install snmp
       -l novlan                 To not install the tibit vlan
   defaults: 
       -d <databaseIpAddress> =  127.0.0.1 
       -n <databaseName>      =  tibit_pon_controller
     
   examples: 
   $0 -e enp0s8.4090 -d 192.168.1.1
   $0 -e enp0s8.4090 -d 192.168.1.1 -c
   $0 -d 192.168.1.1 -m
   $0 -e enp0s8 -l novlan
   $0 -m -s -e enp6s0
   "
   exit 1 # Exit script after printing help
}

printVersion()
# Prints version of script and exits
{
  echo "Version: $VERSION"
  exit 1
}

checkCntlStatus()
# Checks status of PON Controller and exits
{
  STATUS="$(systemctl is-active tibit-poncntl.service)"
  if [ "${STATUS}" = "active" ]; then
      beep
      echo "
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      ERROR: Tibit Pon Controller is currently installed and active.
      "
      systemctl status --no-pager --lines 0 tibit-poncntl.service
      echo "
      If you are upgrading to $VERSION, please run the upgrade script:
      ./upgrade.sh -i <webserver_ip> -d <database_ip> -p <api_port> -n <database_name> -e <eth_interface>
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 1
  fi
}

checkUmtStatus()
# Checks status of UMT Relay and exits
{
  STATUS="$(systemctl is-active tibit-umtrelay.service)"
  if [ "${STATUS}" = "active" ]; then
      beep
      echo "
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      ERROR: Tibit UMT Relay is currently installed and active.
      "
      sudo systemctl status --no-pager --lines 0 tibit-umtrelay.service
      echo "
      If you are upgrading to $VERSION, please run the upgrade script:
      sudo ./upgrade.sh -i <webserver_ip> -d <database_ip> -p <api_port> -n <database_name> -e <eth_interface>
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 1
  fi
  STATUS="$(systemctl is-active tibit-authenticator.service)"
  if [ "${STATUS}" = "active" ]; then
      beep
      echo "
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      ERROR: Tibit UMT Relay is currently installed and active.
      "
      sudo systemctl status --no-pager --lines 0 tibit-authenticator.service
      echo "
      If you are upgrading to $VERSION, please run the upgrade script:
      sudo ./upgrade.sh -i <webserver_ip> -d <database_ip> -p <api_port> -n <database_name> -e <eth_interface>
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 1
  fi
}

checkMgrStatus()
# Checks status of PON Manager
{
  STATUS="$(systemctl is-active apache2.service)"
  if [ "${STATUS}" = "active" ]; then
      beep
      echo "
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      ERROR: Tibit Pon Controller is currently installed and active.
      "
      systemctl status --no-pager --lines 0 apache2.service
      echo "
      If you are upgrading to $VERSION, please run the upgrade script:
      ./upgrade.sh -i <webserver_ip> -d <database_ip> -p <api_port> -n <database_name> -e <eth_interface>
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 1
  fi
}

checkMongoStatus()
# Checks status of MongoDB and exits
{
  STATUS="$(systemctl is-active mongod.service)"
  if [ "${STATUS}" = "active" ]; then
      beep
      echo "
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      ERROR:  MongoDB is currently installed and active.
      "
      systemctl status --no-pager --lines 0 mongod.service
      echo "
      If you are upgrading to $VERSION, please run the upgrade script:
      ./upgrade.sh -i <webserver_ip> -d <database_ip> -p <api_port> -n <database_name> -e <eth_interface>
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 1
  fi
}

checkNetconfStatus()
#n Checks status of NETCONF server and exits
{
  STATUS="$(systemctl is-active tibit-netconf.service)"
  if [ "${STATUS}" = "active" ]; then
      beep
      echo "
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      ERROR: Tibit NETCONF Server is currently installed and active.
      "
      sudo systemctl status --no-pager --lines 0 tibit-netconf.service
      echo "
      If you are upgrading to $VERSION, please run the upgrade script:
      sudo ./upgrade.sh -i <webserver_ip> -d <database_ip> -p <api_port> -n <database_name> -e <eth_interface>
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 1
  fi
}

beep()
# Sends bell character to terminal for audio warning
{
  echo -en "\007"
}

checkIP()
{
  # Function to Validate IP address argument
  ip=$1
  byte1=`echo "$ip"|xargs|cut -d "." -f1`
  byte2=`echo "$ip"|xargs|cut -d "." -f2`
  byte3=`echo "$ip"|xargs|cut -d "." -f3`
  byte4=`echo "$ip"|xargs|cut -d "." -f4`
  
  if [[  $ip =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$  && $byte1 -ge 0 && $byte1 -le 255 && $byte2 -ge 0 && $byte2 -le 255 && $byte3 -ge 0 && $byte3 -le 255 && $byte4 -ge 0 && $byte4 -le 255 ]]
  then
    # Valid IP Address
    return 0
  else
    # Invalid IP Address
    return 1
  fi
}



## Venko Customization: Set custom ONU default firmware download parameters and alarm configuration
setFirmwareDownloadParams()
{
  echo 'Setting Venko Firmware Download Parameters...'
  sed -i 's/"Backoff Divisor": 2,/"Backoff Divisor": 1,/g' /opt/tibit/ponmgr/api/databaseSeedFiles/documents/deviceConfigs/onus/ONU-CFG-Default.json

  sed -i 's/"End Download Timeout": 0,/"End Download Timeout": 60,/g' /opt/tibit/ponmgr/api/databaseSeedFiles/documents/deviceConfigs/onus/ONU-CFG-Default.json
}


setAlarmDefaultConfig()
{
echo 'Setting Venko Default ONU Alarm Configuration...'
sed -i '191c\                "Suppress" : true' /opt/tibit/ponmgr/api/databaseSeedFiles/documents/alarmConfigs/onus/ONU-ALARM-CFG-Default.json
sed -i '187c\                "Suppress" : true' /opt/tibit/ponmgr/api/databaseSeedFiles/documents/alarmConfigs/onus/ONU-ALARM-CFG-Default.json
}
## END Venko Customization: Set custom ONU default firmware download parameters and alarm configuration



## VENKO Customization: Check venko-vlan_setup.service
checkVenkoVlan()
{
  STATUS="$(systemctl  status venko-vlan_setup.service | awk 'END {print $NF}')"
  if [[ "${STATUS}" != "Succeeded." ]]; then
      systemctl status --no-pager --lines 0 venko-vlan_setup.service
      echo "
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

        ERROR: venko-vlan_setup service no installed properly

        Exiting install script

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 2
  fi
}
## END VENKO Customization: Check venko-vlan_setup.service


## VENKO Customization: Check DNS server is ok
checkDNS()
{
  i=10

  while [[ $i -ne 0 ]]
  do
          resolvedIP=$(nslookup "www.google.com" | awk -F':' '/^Address: / { matched = 1 } matched { print $2}' | xargs)
          if [[ -z "$resolvedIP" ]]; then
                  echo "waiting for the DNS server is ok"
                  sleep 3
                  i=$i-1
          else
                  echo "DNS server is ok"
                  break
          fi
  done

  if [[ $i -eq 0 ]]; then
    echo "
    * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    Installation aborted: DNS is not working
    * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    "
    exit 1
  fi
}
## End VENKO Customization: Check DNS server is ok

# Print options
print_options()
{
    echo "
    * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    INSTALL START - OPTIONS:
    DATABASE = ${DATABASE}
    NAME = ${NAME}
    INTERFACE = ${INTERFACE}
    INSTALLTYPE = ${INSTALL_TYPE}
    * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    "
}

# Check to see if root or sudo
if [[ $(id -u) -ne 0 ]] ; then
    beep 
    echo "Please run as root or sudo"
    helpFunction
fi

# Process command line arguments
while getopts :d:n:e:l:s:cmhv opt
do
  case "${opt}" in
    d) DATABASE=${OPTARG};;
    n) NAME=${OPTARG};;
    e) INTERFACE=${OPTARG};;
    c) INSTALLTYPE=c;;
    m) INSTALLTYPE=m;;
    l) NOVLAN=${OPTARG};;
    s) SNMP=true;;
    v) printVersion; exit 1;;
    h) helpFunction;;
    \? ) beep; echo "Unknown option: -$OPTARG" >&2;helpFunction; exit 1;;
    :) beep; echo "Missing option argument for -$OPTARG" >&2; helpFunction; exit 1;;
    *) beep; echo "Unimplemented option: -$opt" >&2; helpFunction; exit 1;;  
  esac
done

# Check command line arguments and set defaults
if [ -z "$INSTALLTYPE" ]; then
    INSTALLTYPE=b
fi
if [ -z "$DATABASE" ]; then
    DATABASE=127.0.0.1
fi
if [ -z "$NAME" ]; then
    NAME=tibit_pon_controller
fi
if [[ -z "$INTERFACE"  && ( $INSTALLTYPE == "c"  || $INSTALLTYPE == "b" ) ]]; then
  beep
  echo "ERROR: -e <l2_eth_interface_name> must be provided for this installation type"
  helpFunction
fi
if [ -z "$NOVLAN" ]; then
   NOVLAN="s4090"
fi

## VENKO Customization: Check DNS server is ok
checkDNS
## End VENKO Customization: Check DNS server is ok

apt-get update -y && apt-get upgrade -y

# PON Manager Install
if [ $INSTALLTYPE == "m" ] || [ $INSTALLTYPE == "b" ]; then
  # Check Manager requirements

  echo
  echo "Checking PON Manager install requirements..."
  if [ -z "$DATABASE" ]; then
      DATABASE=127.0.0.1
  fi

  if [ -z "$NAME" ]; then
      NAME=tibit_pon_controller
  fi

  if ! checkIP ${DATABASE}; then
    beep
    echo; echo "ERROR: Invalid Database IP Address ${DATABASE}"
    echo
    helpFunction
  fi
  
  checkMongoStatus
  checkMgrStatus
  checkNetconfStatus

  echo
  echo "Requirements check successful!"
  echo

echo "

  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

    Installing MongoDB

  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  "
  
  echo "Adding MongoDB public GPG Key and creating list file for MongoDB....."
if [ "${build_os}" == "Ubuntu20.04" ] ; then
  wget -qO - https://www.mongodb.org/static/pgp/server-4.4.asc | apt-key add -
  echo "deb [ arch=amd64,arm64 ] https://repo.mongodb.org/apt/ubuntu focal/mongodb-org/4.4 multiverse" | tee /etc/apt/sources.list.d/mongodb-org-4.4.list
elif [ "${build_os}" == "Ubuntu20.04" ] ; then
  wget -qO - https://www.mongodb.org/static/pgp/server-6.0.asc | apt-key add -
  echo "deb [ arch=amd64,arm64 ] https://repo.mongodb.org/apt/ubuntu focal/mongodb-org/6.0 multiverse" | tee /etc/apt/sources.list.d/mongodb-org-6.0.list
fi
  echo "Updating....." 
  apt-get update
  
  echo "Installing MongoDB and packages....."
  apt-get install -y mongodb-org
  
  echo
  echo "Creating db directories....."
  mkdir /data || true
  mkdir /data/db || true
  echo
  sleep 5
  
  #Venko Customization: configure crontab for logrotate of mongodb
  crontab -l > mycron
  #echo new cron into cron file
  echo "0 1 * * * /usr/sbin/logrotate /etc/logrotate.d/mongod.conf" >> mycron
  #install new cron file
  crontab mycron
  rm mycron
  systemctl enable mongod.service || true
  systemctl start mongod.service
  sleep 5

  # End VENKO Customization: configure crontab for logrotate of mongodb

  # Verify MongoDB service installed correctly and is running.  If not, exit script
  
  STATUS="$(systemctl status mongod.service | grep Loaded: | awk '{ print $2 }')"
  if [[ ! "${STATUS}" = "loaded" ]]; then
      systemctl status --no-pager --lines 0 mongod.service
      beep
      echo "
  
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 
        ERROR: Mongodb was not installed properly
        Verify Internet connectivity and re-try installation
 
        Exiting install script

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 2  
  fi

  #Update mongod.conf for bindIp if needed
  if [ $DATABASE != "127.0.0.1" ]; then
    echo "Updating MongoDB Configuration file with IP address (/etc/mongod.conf)....."
    sed -i '/bindIp:/ c\  bindIp: 127.0.0.1,'$DATABASE /etc/mongod.conf
    echo "Mongo configuration file update complete!"
  fi
   # Update systemctl mongod service to restart on-failure
  sed -i '/^\[Service\]/a Restart=on-failure' /lib/systemd/system/mongod.service
  systemctl daemon-reload || true

  # Enable logRoate for MongoDB
  sed -i '/^  path: \/var\/log\/mongodb\/mongod.log/a\  logRotate: reopen' /etc/mongod.conf
  sed -i '/^processManagement:/a\  pidFilePath: /var/run/mongodb/mongod.pid' /etc/mongod.conf
  mkdir -p /var/run/mongodb
  chown mongodb:mongodb /var/run/mongodb
  mkdir -p /usr/lib/tmpfiles.d
  echo "d /var/run/mongodb 0755 mongodb mongodb" | tee /usr/lib/tmpfiles.d/mongodb.conf

  echo "/var/log/mongodb/mongod.log {
  size 100M
  rotate 4
  missingok
  compress
  delaycompress
  notifempty
  create 640 mongodb mongodb
  sharedscripts
  postrotate
    /bin/kill -SIGUSR1 \$(cat /var/run/mongodb/mongod.pid)
  endscript
}" | tee /etc/logrotate.d/mongod.conf

  #VENKO Customization: Create users in MongoDB
  #Enable the authentication for mongodb.
echo "Creating MongoDB users....."
echo "
db.createUser(
{
user: \"admin\",
pwd: \"V3nk0-2021!\",
roles: [ { role: \"root\", db: \"admin\" } ]

}
)
" > mongo_admin.js

  mongo --quiet admin mongo_admin.js > /dev/null
  sleep 5

echo "
db.createUser(
{
user: \"pdmPonController\",
pwd: \"V3nk0Con\",
roles: [ { role: \"readWrite\", db: \"tibit_pon_controller\" } ]
}
)
db.createUser(
{
user: \"pdmPonManager\",
pwd: \"V3nk0Man\",
roles: [ { role: \"readWrite\", db: \"tibit_pon_controller\" },
{ role: \"readWrite\", db: \"tibit_users\" } ]
}
)
db.createUser(
{
user: \"pdmUserAdmin\",
pwd: \"v3nk0Admin\",
roles: [ { role: \"readWrite\", db: \"tibit_users\" } ]
}
)
db.createUser(
{
user: \"pdmNetconf\",
pwd: \"V3nk0Net\",
roles: [ { role: \"readWrite\", db: \"tibit_pon_controller\" } ,
{ role: \"readWrite\", db: \"tibit_netconf\" } ]
}
)
" > mongo.js
  mongo --quiet tibit_users mongo.js  > /dev/null
  sleep 5

  rm *.js

  #Creating key file to use Authentication
  echo "Creating MongoDB keyfile for Authentication....."
  openssl rand -base64 756 > mongodb-keyfile
  mv mongodb-keyfile /etc/ssl
  chown mongodb:mongodb /etc/ssl/mongodb-keyfile
  chmod 400 /etc/ssl/mongodb-keyfile
  sleep 5

  #End VENKO Customization: Create users in MongoDB


  # Enable and initiate replSet in Mongo (needed for Netconf)
  echo "Updating MongoDB configuration file with Replication data (/etc/mongod.conf)....."
  sed -i '/replication/ c\replication: \n  replSetName: \"rs0\"\n' /etc/mongod.conf
  echo "Mongo configuration file update complete!"
  echo 
  echo "Starting MongoDB....."
  systemctl enable mongod.service || true
  systemctl restart mongod.service || true
  
  sleep 5
  # Verify MongoDB service is running.  If not, exit script
  
  STATUS="$(systemctl is-active mongod.service)"
  if [[ ! "${STATUS}" = "active" ]]; then
      systemctl status --no-pager --lines 0 mongod.service
      beep
      echo "
  
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

        ERROR: Mongodb configuration file \"/etc/mongod.conf\" not installed properly

        Exiting install script

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 2  
  fi
  echo "MongoDB started successfully."
  
  # Wait until Mongo is ready to accept connections, exit if this does not happen within 10 seconds
  COUNTER=0
  until mongo --host ${DATABASE} --port ${DATABASE_PORT} --eval "printjson(db.serverStatus())" >& /dev/null
  do
    sleep 1
    COUNTER=$((COUNTER+1))
    if [[ ${COUNTER} -eq 10 ]]; then
      beep
      echo "
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

        ERROR: MongoDB server $DATABASE:$DATABASE_PORT is not responding

        Exiting install script

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 2
    fi
    echo "Waiting for MongoDB server $DATABASE:$DATABASE_PORT to respond... ${COUNTER}/10"
  done
  
  #log into mongocli  and turn on replication  
  echo "rs.initiate()" | mongo $DATABASE:$DATABASE_PORT --quiet >& /dev/null
  sleep 5
  
  # Check that replication is inititated in MongoDB
  COMMAND="db.system.replset.find()"
  RESP_RSFIND="$(echo $COMMAND | mongo $DATABASE:$DATABASE_PORT/local --quiet | awk '{ print $4 }')"
  if [ "$RESP_RSFIND" != "\"rs0\"," ]; then
      beep
      echo "
  
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

        ERROR: MongoDB replSet rs0 not intialized properly.

        Exiting install script

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
    exit 2
  fi

  ## VENKO Customization: Enable and initiate mongo authentication
  echo "
security:
  keyFile: /etc/ssl/mongodb-keyfile
" >> /etc/mongod.conf
  systemctl restart mongod.service
  sleep 5
  ## END VENKO Customization: Enable and initiate mongo authentication

  echo "MongoDB Install Complete!"
  
  # Installation of PON Manager
  echo "

  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

    Installing MCMS PON Manager $PONMGRVER

  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

  $INITDIR/$PONMGRVER-PonManager-$PLATFORM/
  Executing: apt-get install ./$ManagerImage -y
  "
  cd $INITDIR/$PONMGRVER-PonManager-$PLATFORM/
  apt-get install ./$ManagerImage -y

  ##VENKO Customization: Create DJANGOKEY

  chmod 777 ./tools/generate_django_key.py
  DJANGOKEY=$(./tools/generate_django_key.py)

  ##End VENKO Customization: Create DJANGOKEY

  cd $INITDIR

  ##VENKO Customization: Edit users, passwords and enable authentication
  echo
  echo "Creating configuration files"
  echo "
{
    \"Default\": {
        \"auth_enable\": true,
        \"auth_db\": \"tibit_users\",
        \"tls_enable\": false,
        \"ca_cert_path\": \"\",
        \"host\": \"$DATABASE\",
        \"name\": \"$NAME\",
        \"username\": \"pdmPonManager\",
        \"password\": \"V3nk0Man\",
        \"port\": \"27017\",
        \"replica_set_enable\": false,
        \"replica_set_name\": \"rs0\",
        \"replica_set_hosts\": [
            \"$DATABASE:27017\"
        ]
    }
}
" > /etc/tibit/ponmgr/databases.json

  echo "
{
    \"auth_enable\": true,
    \"auth_db\": \"tibit_users\",
    \"tls_enable\": false,
    \"ca_cert_path\": \"\",
    \"dns_srv\": false,
    \"host\": \"$DATABASE\",
    \"name\": \"tibit_users\",
    \"username\": \"pdmUserAdmin\",
    \"password\": \"v3nk0Admin\",
    \"port\": \"27017\",
    \"db_uri\": \"\",
    \"replica_set_enable\": false,
    \"replica_set_name\": \"rs0\",
    \"replica_set_hosts\": [
        \"$DATABASE:27017\"
    ],
    \"django_key\": \"$DJANGOKEY\",
    \"compression\":false
}
" > /etc/tibit/ponmgr/user_database.json
  ##END VENKO Customization: Edit users, passwords and enable authentication
  echo
  echo "Starting MCMS PON Manager"
  systemctl restart apache2.service
  sleep 5
  
  #Check Apache2 (PON Manager) status
  STATUS="$(systemctl is-active apache2.service)"
  if [[ ! "${STATUS}" = "active" ]]; then
      systemctl status --no-pager --lines 0 apache2.service
      beep
      echo "
  
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

        ERROR: PON Manager (Apache2) was not installed properly

        Exiting install script

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 2  
  fi
  echo "PON Manager install complete!"

  ## Venko Customization: Set custom ONU default firmware download parameters and alarm configuration
  setFirmwareDownloadParams
  setAlarmDefaultConfig
  ## End Venko Customization: Set custom ONU default firmware download parameters and alarm configuration

  ## VENKO Customization: Copy files of venko
  echo "Venko database files...."
  cp logos/*.png /opt/tibit/ponmgr/ponmgr/assets/branding
  cp footer/*.txt /opt/tibit/ponmgr/ponmgr/assets/branding

  echo "Service Configs DB"
  rm -f /opt/tibit/ponmgr/api/databaseSeedFiles/documents/srvConfigs/*
  cp databaseSeedFiles/srvConfigs/*.json /opt/tibit/ponmgr/api/databaseSeedFiles/documents/srvConfigs/
  chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/documents/srvConfigs/*
  cp databaseSeedFiles/firmwares/olts/* /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/olts
  mkdir -p /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/onus
  cp databaseSeedFiles/firmwares/onus/* /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/onus
  chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/onus/*
  chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/olts/*
  chown www-data:www-data /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/onus/*
  chown www-data:www-data  /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/olts/*
  rm -f /opt/tibit/ponmgr/api/databaseSeedFiles/documents/pictures/onus/*
  rm -f /opt/tibit/ponmgr/api/databaseSeedFiles/documents/pictures/olts/*
  rm -f /opt/tibit/ponmgr/api/databaseSeedFiles/documents/pictures/controllers/*
  cp databaseSeedFiles/pictures/olts/* /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/olts/
  cp databaseSeedFiles/pictures/onus/* /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/onus/
  cp databaseSeedFiles/pictures/controllers/* /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/controllers/
  chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/controllers/*
  chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/onus/*
  chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/olts/*
  chown www-data:www-data  /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/controllers/*
  chown www-data:www-data /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/onus/*
  chown www-data:www-data /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/olts/*
  cp metadata.json /opt/tibit/ponmgr/api/databaseSeedFiles/
  chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/metadata.json

  echo "Venko Netconf Scripts"
  mkdir -p  /opt/tibit/netconf/examples/venko
  cp -R $INITDIR/$NETCONFVER-Netconf-ConfD-$PLATFORM/examples/venko/* /opt/tibit/netconf/examples/venko
  ## END VENKO Customization: Copy files of venko

fi
# END Pon Manager install

# Installation of MCMS Netconf server

  echo "
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

    Installing MCMS NETCONF ConfD Server $NETCONFVER

  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  
  $INITDIR/$NETCONFVER-Netconf-ConfD-$PLATFORM/
  Executing: apt-get install ./$NetconfImage -y
  "
  cd $INITDIR/$NETCONFVER-Netconf-ConfD-$PLATFORM/
  apt-get install ./$NetconfImage -y
  cd $INITDIR
  sleep 5

  sed -i '/\"host\":/ c\        \"host\": \"'$DATABASE'\",' /etc/tibit/netconf/NetconfInit.json 
  sed -i '/\"name\":/ c\        \"name\": \"'$NAME'\",' /etc/tibit/netconf/NetconfInit.json 
  sed -i '/\"auth_enable\":/ c\        \"auth_enable\": true,' /etc/tibit/netconf/NetconfInit.json 
  sed -i '/\"username\":/ c\        \"username\": \"pdmNetconf\",' /etc/tibit/netconf/NetconfInit.json 
  sed -i '/\"password\":/ c\        \"password\": \"V3nk0Net\",' /etc/tibit/netconf/NetconfInit.json 

## VENKO Customization: install python3-ncclient
apt-get install -y python3-ncclient
## END VENKO Customization: install python3-ncclient

## VENKO Customization: Create user to netconf in mongoDB


systemctl restart tibit-netconf.service

systemctl status --no-pager --lines 0 apache2.service
systemctl status --no-pager --lines 0 mongod.service
systemctl status --no-pager --lines 0 tibit-*.service
## END VENKO Customization: Create user to netconf in mongoDB

#Check NETCONF Install status
STATUS="$(systemctl is-active tibit-netconf.service)"
if [[ ! "${STATUS}" = "active" ]]; then
    systemctl status --no-pager --lines 0 tibit-netconf.service
    beep
    echo "

    * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

      ERROR: NETCONF Service was not installed properly

      Exiting install script

    * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    "
    exit 2  
fi
  # Enabling user for access to Netconf server
  LOGIN_NAME="$(logname)"
  /opt/tibit/netconf/examples/nacm/nacm-config.sh /opt/tibit/netconf/examples/nacm/nacm-example.xml
  /opt/tibit/netconf/examples/nacm/nacm-config.sh netconf-user.xml
  /opt/tibit/netconf/examples/nacm/nacm-add-user.sh ${LOGIN_NAME} admin

  systemctl restart tibit-netconf.service
  sleep 5

echo "NETCONF Server install complete!"
systemctl status --no-pager --lines 0 tibit-netconf.service

echo
echo "Checking MCMS PON Manager status....."
echo
systemctl status --no-pager --lines 0 apache2.service
systemctl status --no-pager --lines 0 mongod.service

echo "MCMS PON Manager install completed successfully."  

## VENKO Customization: Install SNMP
if [[ "$INSTALLTYPE" = "b" || "$SNMP" = "true" ]]; then
  echo "

  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    Installing Tibit-Venko SNMP $SNMPVERSION TOOL
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

  Executing: apt-get install ./$SnmpImage -y
  "
  cd $INITDIR/$SNMPVERSION-Snmp/
  echo "$DATABASE" | sudo apt-get install -y ./$SnmpImage
  cd $INITDIR
fi
## END VENKO Customization: Install SNMP

# PON Controller install
if [ $INSTALLTYPE == "c" ] || [ $INSTALLTYPE == "b" ]; then
  echo
  echo "Checking PON Controller install requirements..."    

  # Check Controller requirements
  if [ -z "$DATABASE" ]; then
      DATABASE=127.0.0.1
  fi

  if [ -z "$NAME" ]; then
      NAME=tibit_pon_controller
  fi

  if [ -z "$INTERFACE" ]; then
      beep
      echo "No valid interface specified. Use -e <l2_eth_interface> and specify ethernet interface to PON (e.g. ens192)"
      helpFunction
  fi

  if ! checkIP ${DATABASE}; then
    beep
    echo; echo "ERROR: Invalid Database IP Address ${DATABASE}"
    echo
    helpFunction
  fi

  checkCntlStatus
  
  checkUmtStatus

  ## VENKO Customization: create vlan on interface selected to communicate with TIBIT
  echo "

  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

    Installing JQ and configuring Vlan
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  "
  sudo apt-get install -y jq

  if [ "$NOVLAN" = "s4090" ]; then
    echo "
    * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    Activating venko-vlan_setup.service
    * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    "
    mkdir -p /opt/tibit/
    echo "#!/bin/bash

ip link set ${INTERFACE} up
ip link add link ${INTERFACE} name ${INTERFACE}.s4090 type vlan id 4090 egress-qos-map 0:7 1:7 2:7 3:7 4:7 5:7 6:7
ip link set dev ${INTERFACE}.s4090 up
" > /opt/tibit/setup-vlan.sh
    chmod +x /opt/tibit/setup-vlan.sh
    /opt/tibit/setup-vlan.sh
    echo "[Unit]
Description=Venko Create vlan interface for Poncntl
After=network-online.target
Wants=network-online.target
Before=tibit-poncntl.service

[Service]
ExecStart=/opt/tibit/setup-vlan.sh
Type=exec
Restart=no

[Install]
WantedBy=user.target
RequiredBy=tibit-poncntl.service
" > /etc/systemd/system/venko-vlan_setup.service
    sudo systemctl daemon-reload
    sudo systemctl enable venko-vlan_setup.service
    sudo systemctl start venko-vlan_setup.service
    checkDNS
    TIBITINTERFACE="$INTERFACE.s4090"
    sleep 10
  else
    echo "Installing Tibit without use vlan 4090"
    TIBITINTERFACE=$INTERFACE
  fi
  ## End VENKO Customization: create vlan on interface selected to communicate with TIBIT

  echo
  echo "Requirements check successful!"
  echo

  # Installation of PON Controller
  
  echo "
  
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

    Installing PON Controller $PONCNTLVER with L2 interface $INTERFACE

  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  
  $INITDIR/$PONCNTLVER-PonController-$PLATFORM-amd64/
  Executing: apt-get install ./$ControllerImage -y
  "
  cd $INITDIR/$PONCNTLVER-PonController-$PLATFORM-amd64/
  apt-get install ./$ControllerImage -y
  cd $INITDIR

  ## VENKO Customization: remove tibit SRV*.json and edit venko configuring file
  echo "Removing old SRV"
  rm -f /opt/tibit/poncntl/database/SRV*.json
  
  echo "Creating configuration file..."
echo "
{
    \"CNTL\": {
        \"Auth\": false, 
        \"CFG Version\": \"$PONCNTLVER\", 
        \"DHCPv4\": true,
        \"DHCPv6\": true,
        \"PPPoE\": true,
        \"UMT interface\": \"tibitvirt\"
    }, 
    \"DEBUG\": {}, 
    \"JSON\": {
        \"databaseDir\": \"/opt/tibit/poncntl/database/\", 
        \"defaultDir\": \"/opt/tibit/poncntl/database/\"
 }, 
    \"Local Copy\": {
        \"CNTL-STATE\": false, 
        \"OLT-STATE\": false, 
        \"ONU-STATE\": false
    }, 
    \"Logging\": {
        \"Directory\": \"/var/log/tibit\", 
        \"FileCount\": 3, 
        \"FileSize\": 1024000,
        \"Tracebacks\": true
    }, 
    \"MongoDB\": {
      \"auth_db\": \"tibit_users\",
      \"auth_enable\": true,
      \"ca_cert_path\": \"/etc/tibit/ca.pem\",
      \"host\": \"$DATABASE\",
      \"name\": \"$NAME\",
      \"password\": \"V3nk0Con\",
      \"port\": \"27017\",
      \"tls_enable\": false,
      \"username\": \"pdmPonController\",
      \"dns_srv\": false,
      \"db_uri\": \"\",
      \"replica_set_enable\": false,
      \"replica_set_name\": \"rs0\",
      \"replica_set_hosts\":
          [
          \"10.10.10.5:27017\",
          \"10.20.10.20:27999\",
          \"mongo02.example.com:17999\"
          ],
      \"valicate_cfg\": true
  },
    \"databaseType\": \"MongoDB\", 
    \"interface\": \"$TIBITINTERFACE\"
}
" > /etc/tibit/poncntl/PonCntlInit.json
  ## END VENKO Customization: remove tibit SRV*.json and edit vanko configuring filei

# Try to Populate MongoDB with default database files.
# If return in <> 0 then output error message

if [ -d /opt/tibit/poncntl/database ]; then
  echo "
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 

  Attempting to populate MongoDB default database.
  A working connection to the MongoDB server is required.

  Press ^c to abort.

  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *  
  "
  /opt/tibit/poncntl/bin/JsonToMongo -c /etc/tibit/poncntl/PonCntlInit.json
  retVal=$?
  if [ $retVal -ne 0 ]; then
    echo "
    
    * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 

    Error: Unable to populate default MongoDB database $NAME.
    You may need to rerun JsonToMongo script.

    * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
    "
  fi

  ## Venko Customization: Check venko-vlan_setup.service
  if [[ "$VLAN" =  "4090" ]]; then
    checkVenkoVlan
  fi
  ## End Venko Customization: Check venko-vlan_setup.service

fi

# Start PON Controller
  sleep 2
  echo "Starting tibit-poncntl....."
  systemctl restart tibit-poncntl.service
  sleep 2

  #Check PON Controller service install status
  STATUS="$(systemctl is-active tibit-poncntl.service)"
  if [[ ! "${STATUS}" = "active" ]]; then
      systemctl status --no-pager --lines 0 tibit-poncntl.service
      beep
      echo "
  
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

        ERROR: PON Controller was not installed properly

        Exiting install script

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 2  
  fi
  #Check UMT Relay service install status
  STATUS="$(systemctl is-active tibit-umtrelay.service)"
  if [[ ! "${STATUS}" = "active" ]]; then
      systemctl status --no-pager --lines 0 tibit-umtrelay.service
      beep
      echo "
  
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

        ERROR: UMT Relay was not installed properly

        Exiting install script

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 2  
  fi
  echo "PON Controller install complete."
  
  systemctl status --no-pager --lines 0 tibit-poncntl.service
  systemctl status --no-pager --lines 0 tibit-umtrelay.service

  echo "Status of Tibit Services....."
  systemctl status --no-pager --lines 0 apache2.service
  systemctl status --no-pager --lines 0 mongod.service
  systemctl status --no-pager --lines 0 tibit-*.service
#  systemctl status --no-pager --lines 0 snmp-tibit.service

  echo "
  
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      Below is a list of commands for administering the PON Controller software.
  
      Use the following command to start the PON Controller:
  
          systemctl start tibit-poncntl
  
      Use the following command to restart the PON Controller:
  
          systemctl restart tibit-poncntl
  
      Use the following command to stop the PON Controller:
  
          systemctl stop tibit-poncntl
  
      Use the following command to check the status of the PON Controller:
  
          systemctl status tibit-poncntl
  
      Use the following command to view log messages generated by the PON Controller process:
  
          tail -f /var/log/tibit/ponCntl.log
  
      Use the following command to display the most recent console messages logged by
      the PON Controller process:
  
          journalctl -r -u tibit-poncntl
  
      Use the following command to \"tail\" console messages logged by the PON Controller
      process (similar to tail -f):
  
          journalctl -f -u tibit-poncntl
  
      Note: The systemd journaling system logs both console messages (stdout and stderr) and
      Syslog messages from PON Controller process by default. Disable either Console or Syslog
      logging in the PON Controller Configuration in MongoDB (CNTL-CFG) to avoid the duplicate
      log messages.
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  
  "
fi

echo
echo "Installation complete!"
echo

#Retrieve and print MCMS Package installed versions
PonMgrVersionInstalled="Not Installed"
PonCntlVersionInstalled="Not Installed"
PonNetconfVersionInstalled="Not Installed"

PonMgrVersionInstalled="$(apt-cache policy tibit-ponmgr | grep Installed | awk '{ print $2 }' | awk -F: '{ print $2 }')"
PonCntlVersionInstalled="$(apt-cache policy tibit-poncntl | grep Installed | awk '{ print $2 }' | awk -F: '{ print $2 }')"
PonNetconfVersionInstalled="$(apt-cache policy tibit-netconf | grep Installed | awk '{ print $2 }' | awk -F: '{ print $2 }')"

if [ -z $PonMgrVersionInstalled ] || [ $PonMgrVersionInstalled == "" ] || [ $PonMgrVersionInstalled == " " ]; then
  PonMgrVersionInstalled="Not Installed"
else
  PonMgrVersionInstalled="R$PonMgrVersionInstalled"
fi
if [ -z $PonCntlVersionInstalled ] || [ $PonCntlVersionInstalled == "" ] || [ $PonCntlVersionInstalled == " " ]; then
  PonCntlVersionInstalled="Not Installed"
else
  PonCntlVersionInstalled="R$PonCntlVersionInstalled"
fi
if [ -z $PonNetconfVersionInstalled ] || [ $PonNetconfVersionInstalled == "" ] || [ $PonNetconfVersionInstalled == " " ]; then
  PonNetconfVersionInstalled="Not Installed"
else
  PonNetconfVersionInstalled="R$PonNetconfVersionInstalled"
fi

echo "MCMS Component Versions:
     PON Manager: $PonMgrVersionInstalled
     PON NETCONF: $PonNetconfVersionInstalled
  PON Controller: $PonCntlVersionInstalled
  "

## VENKO Customization: Configure logrotate to hourly and size 500MB for syslogs
cp rsyslog /etc/logrotate.d/
cp /etc/cron.daily/logrotate /etc/cron.hourly/
cp logrotate.conf /etc/
## End Customization: Configure logrotate to hourly and size 500MB for syslogs
