#!/usr/bin/env bash
VERSION=R5.2.0
INITDIR=$(pwd)
PONCNTLVER=R5.2.0
NETCONFVER=R5.2.0
PONMGRVER=R5.2.0
DATABASE_PORT=27017
PLATFORM=UB2004
DATABASE=127.0.0.1
NAME=tibit_pon_controller
LogFile=finish_install.log
NetconfImage=tibit-netconf-confd_${NETCONFVER}_amd64.deb


exec > >( tee -a "${LogFile}" )
exec 2> >( tee -a "${LogFile}" >&2 )


setFirmwareDownloadParams()
{
echo 'Setting Venko Firmware Download Parameters...'
sed -i 's/"Backoff Divisor": 2,/"Backoff Divisor": 1,/g' /opt/tibit/ponmgr/api/databaseSeedFiles/documents/deviceConfigs/onus/ONU-CFG-Default.json

sed -i 's/"End Download Timeout": 0,/"End Download Timeout": 60,/g' /opt/tibit/ponmgr/api/databaseSeedFiles/documents/deviceConfigs/onus/ONU-CFG-Default.json
}

setAlarmDefaultConfig()
{
echo 'Setting Venko Default ONU Alarm Configuration...'
sed -i '191c\                "Suppress" : true' /opt/tibit/ponmgr/api/databaseSeedFiles/documents/alarmConfigs/onus/ONU-ALARM-CFG-Default.json
sed -i '187c\                "Suppress" : true' /opt/tibit/ponmgr/api/databaseSeedFiles/documents/alarmConfigs/onus/ONU-ALARM-CFG-Default.json
}

askForInterface()
{
	INTER=()
	i=0
	echo -e "Select a Interface for OLT Connection:\n"
	for IF in $(ip link show | awk -F: '$1>0 {print $2}')
	do
		    if [ "$IF" != "lo" ]; then
		            i=$((i+1))
		            IP=$(ip addr show dev "$IF" | awk '$1=="inet"{print gensub("/.*","",1,$2)}' | xargs)
		            INTER+=("$IF");
		            echo -e "$i) Name: $IF  \tAddress: $IP";
		    fi
	done
	is_valid()
	{
		a=$1
		b=$2
		if [ "$a" -eq "$a" ] 2>/dev/null && [ "$a" != "0" ] ; then
				if [ $a -le $b ]; then
				        return 0;
				else
				        return 1;
				fi
		else
				return 1;
		fi
	}

	opt=100
	while ! is_valid "$opt" "$i"
	do
		    read -sn1  opt
	done
	opt=$((opt-1))
	INTER="${INTER[$opt]}"
	echo -e "Tibit will use interface $INTER to connect to the OLT\n".
	return 0
}


askForVlan()
{
	echo -e "Tibit uses VLAN 4090 to connect to olt on interface $INTER\n"
	while true
	do
	 read -r -p "Do you want use vlan 4090? [Y/n] " input
	 case $input in
		 [yY][eE][sS]|[yY])
	 	 VLAN="4090"
		 echo -e "The OLT will be conntect to $INTER:s$VLAN.\n"
	 break
	 ;;
		 [nN][oO]|[nN])
  	 	 VLAN="NOVLAN"
		 echo -e "The OLT will be conntect to $INTER without VLAN.\n"
	 break
			;;
			 *)
	 echo "Invalid input..."
	 ;;
	 esac
	done
	return 0
}

askForMongoIP()
{
        echo -e "MongoDB is configured to use IP $DATABASE and port $DATABASE_PORT\n"
        while true
        do
         read -r -p "Do you want to change IP? [Y/n] " input
         case $input in
                 [yY][eE][sS]|[yY])
		 read -r -p "Please enter a valid IP:" DATABASE
		 while true
		 do
			if ! checkIP ${DATABASE}; then
				echo -e "ERROR: Invalid Database IP Address ${DATABASE}.\n"
				read -r -p "Please enter a valid IP:" DATABASE
			else
			     	echo -e "MongoDB will use IP ${DATABASE}.\n"
				break
			fi
		done
         break
         ;;
                 [nN][oO]|[nN])
                 echo -e "The MongoDB will use the Default IP.\n"
         break
                        ;;
                         *)
         echo "Invalid input..."
         ;;
         esac
        done


        while true
        do
         read -r -p "Do you want to change PORT? [Y/n] " input
         case $input in
                 [yY][eE][sS]|[yY])
                 read -r -p "Please enter a valid PORT:" DATABASE_PORT
         break
         ;;
                 [nN][oO]|[nN])
                 echo -e "The MongoDB will use the Default PORT.\n"
         break
                        ;;
                         *)
         echo "Invalid input..."
         ;;
         esac
        done

	echo -e "The MongoDB will be accessible using  ${DATABASE}:${DATABASE_PORT}\n"
        return 0
}

checkIP()
{
  # Function to Validate IP address argument
  ip=$1
  byte1=`echo "$ip"|xargs|cut -d "." -f1`
  byte2=`echo "$ip"|xargs|cut -d "." -f2`
  byte3=`echo "$ip"|xargs|cut -d "." -f3`
  byte4=`echo "$ip"|xargs|cut -d "." -f4`
  
  if [[  $ip =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$  && $byte1 -ge 0 && $byte1 -le 255 && $byte2 -ge 0 && $byte2 -le 255 && $byte3 -ge 0 && $byte3 -le 255 && $byte4 -ge 0 && $byte4 -le 255 ]]
  then
    # Valid IP Address
    return 0
  else
    # Invalid IP Address
    return 1
  fi
}

checkPonControllerRunningOK()
{
  STATUS="$(systemctl is-active tibit-poncntl.service)"
  if [[ ! "${STATUS}" = "active" ]]; then
      systemctl status --no-pager --lines 0 tibit-poncntl.service
      echo "
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

        ERROR: PON Controller was not installed properly

        Exiting install script

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 2
  fi
}

checkNetConfRunningOK()
{
STATUS="$(systemctl is-active tibit-netconf.service)"
if [[ ! "${STATUS}" = "active" ]]; then
    systemctl status --no-pager --lines 0 tibit-netconf.service
    echo "

     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

       ERROR: NETCONF Service was not installed properly

       Exiting install script

     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     "
     exit 2
fi
}

checkMongoRunningOK()
{
  STATUS="$(systemctl status mongod.service | grep Loaded: | awk '{ print $2 }')"
  if [[ ! "${STATUS}" = "loaded" ]]; then
      systemctl status --no-pager --lines 0 mongod.service
      echo "

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
        ERROR: Mongodb was not installed properly

        Exiting install script

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 2
    fi
echo "mongodb is running"
}

checkApacheRunningOK(){
STATUS="$(systemctl is-active apache2.service)"
if [[ ! "${STATUS}" = "active" ]]; then
      systemctl status --no-pager --lines 0 apache2.service
      echo "

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

        ERROR: PON Manager (Apache2) was not installed properly

        Exiting install script

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 2
fi
}

checkVenkoVlanRunningOK()
{
  STATUS="$(systemctl  status venko-vlan_setup.service | awk 'END {print $NF}')"
  if [[ "${STATUS}" != "Succeeded." ]]; then
      systemctl status --no-pager --lines 0 venko-vlan_setup.service
      echo "
      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

        ERROR: venko-vlan_setup service no installed properly

        Exiting install script

      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
      "
      exit 2
  fi
}


configMongo()
{
  echo "Creating MongoDB directories....."

  mkdir /data || true
  mkdir /data/db || true
  echo
  sleep 5
  echo "Setting Log Rotate for MongoDB..."
  mkdir  -p /mongodb/ || true
  touch /mongodb/mongodb.pid
  chown -R  mongodb:mongodb /mongodb/
  cp mongod.conf /etc/logrotate.d/
  sed -i '/^  path: \/var\/log\/mongodb\/mongod.log/a\  logRotate: reopen' /etc/mongod.conf
  sed -i '/^processManagement:/a\  pidFilePath: /var/run/mongodb/mongod.pid' /etc/mongod.conf
  mkdir -p /var/run/mongodb
  chown mongodb:mongodb /var/run/mongodb
  mkdir -p /usr/lib/tmpfiles.d
  echo "d /var/run/mongodb 0755 mongodb mongodb" | tee /usr/lib/tmpfiles.d/mongodb.conf

  echo "/var/log/mongodb/mongod.log {
    size 100M
    rotate 4
    missingok
    compress
    delaycompress
    notifempty
    create 640 mongodb mongodb
    sharedscripts
    postrotate
      /bin/kill -SIGUSR1 \$(cat /var/run/mongodb/mongod.pid)
    endscript
  }" | tee /etc/logrotate.d/mongod.conf
  crontab -l > mycron
  #echo new cron into cron file
  echo "0 1 * * * /usr/sbin/logrotate /etc/logrotate.d/mongod.conf" >> mycron
  #install new cron file
  crontab mycron
  rm mycron
  echo "Enable mongo service"
  systemctl enable mongod.service || true
  echo "Start mongodb"

  systemctl restart mongod.service
  sleep 10

  checkMongoRunningOK

  if [ $DATABASE != "127.0.0.1" ]; then
      echo "Updating MongoDB Configuration file with IP address (/etc/mongod.conf)....."
      sed -i '/bindIp:/ c\  bindIp: 127.0.0.1,'$DATABASE /etc/mongod.conf
      echo "Mongo configuration file update complete!"
  fi
  # Update systemctl mongod service to restart on-failure
  sed -i '/^\[Service\]/a Restart=on-failure' /lib/systemd/system/mongod.service
  systemctl daemon-reload || true
  sleep 5

  checkMongoRunningOK

  echo "Creating MongoDB users....."
  echo "
  db.createUser(
  {
  user: \"admin\",
  pwd: \"V3nk0-2021!\",
  roles: [ { role: \"root\", db: \"admin\" } ]

  }
  )
  " > mongo_admin.js
  sleep 5

  checkMongoRunningOK


  mongo --quiet admin mongo_admin.js > /dev/null
  echo "user admin created on mongdb"
  sleep 5

  echo "
  db.createUser(
  {
  user: \"pdmPonController\",
  pwd: \"V3nk0Con\",
  roles: [ { role: \"readWrite\", db: \"tibit_pon_controller\" } ]
  }
  )
  db.createUser(
  {
  user: \"pdmPonManager\",
  pwd: \"V3nk0Man\",
  roles: [ { role: \"readWrite\", db: \"tibit_pon_controller\" },
  { role: \"readWrite\", db: \"tibit_users\" } ]
  }
  )
  db.createUser(
  {
  user: \"pdmUserAdmin\",
  pwd: \"v3nk0Admin\",
  roles: [ { role: \"readWrite\", db: \"tibit_users\" } ]
  }
  )
  db.createUser(
  {
  user: \"pdmNetconf\",
  pwd: \"V3nk0Net\",
  roles: [ { role: \"readWrite\", db: \"tibit_pon_controller\" } ,
  { role: \"readWrite\", db: \"tibit_netconf\" } ]

  }
  )
  " > mongo.js
  sleep 5


  checkMongoRunningOK

  mongo --quiet tibit_users mongo.js > /dev/null
  echo "tibit users created on mongdb"
  sleep 5
  rm *.js

  #Creating key file to use Authentication
  echo "Creating MongoDB keyfile for Authentication....."
  openssl rand -base64 756 > mongodb-keyfile
  mv mongodb-keyfile /etc/ssl
  chown mongodb:mongodb /etc/ssl/mongodb-keyfile
  chmod 400 /etc/ssl/mongodb-keyfile
  sleep 5

  echo "Updating MongoDB configuration file with Replication data (/etc/mongod.conf)....."
  sed -i '/replication/ c\replication: \n  replSetName: \"rs0\"\n' /etc/mongod.conf
  echo "Mongo configuration file update complete!"
  echo
  echo "Starting MongoDB....."
  systemctl enable mongod.service || true
  systemctl restart mongod.service
  sleep 5

  checkMongoRunningOK

  echo "MongoDB checking connection...."

  COUNTER=0
  until mongo --host ${DATABASE} --port ${DATABASE_PORT} --eval "printjson(db.serverStatus())" >& /dev/null
  do
      sleep 3
      COUNTER=$((COUNTER+1))
      if [[ ${COUNTER} -eq 10 ]]; then
          echo "
          * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

              ERROR: MongoDB server $DATABASE:$DATABASE_PORT is not responding
              Exiting install script

          * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *"
          exit 2
      fi
      echo "Waiting for MongoDB server $DATABASE:$DATABASE_PORT to respond... ${COUNTER}/10"
  done

  echo "MongoDB connection is ok."
  echo "Checking baisc replicaset configuration."

  echo "rs.initiate()" | mongo $DATABASE:$DATABASE_PORT
  sleep 5
  # Check that replication is inititated in MongoDB
  COMMAND="db.system.replset.find()"
  RESP_RSFIND="$(echo $COMMAND | mongo $DATABASE:$DATABASE_PORT/local --quiet | awk '{ print $4 }')"
  if [ "$RESP_RSFIND" != "\"rs0\"," ]; then
      echo "
          * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
          ERROR: MongoDB replSet rs0 not intialized properly.
          Exiting install script
          * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *"
      exit 2
    fi

  echo "MongoDB Replicaset configuration is ok."

  echo "
security:
  keyFile: /etc/ssl/mongodb-keyfile
  " >> /etc/mongod.conf
  systemctl restart mongod.service
  sleep 5

  checkMongoRunningOK

  echo "MongoDB Configuration Complete!"
}


configPonManager()
{
cd $INITDIR/$PONMGRVER-PonManager-$PLATFORM/
chmod 777 ./tools/generate_django_key.py
DJANGOKEY=$(./tools/generate_django_key.py)
cd $INITDIR
echo "Creating configuration files"

echo "
{
    \"Default\": {
        \"auth_enable\": true,
        \"auth_db\": \"tibit_users\",
        \"tls_enable\": false,
        \"ca_cert_path\": \"\",
        \"host\": \"$DATABASE\",
        \"name\": \"$NAME\",
        \"username\": \"pdmPonManager\",
        \"password\": \"V3nk0Man\",
        \"port\": \"27017\",
        \"replica_set_enable\": false,
        \"replica_set_name\": \"rs0\",
        \"replica_set_hosts\": [
            \"$DATABASE:27017\"
        ]
    }
}
" > /etc/tibit/ponmgr/databases.json

echo "
{
    \"auth_enable\": true,
    \"auth_db\": \"tibit_users\",
    \"tls_enable\": false,
    \"ca_cert_path\": \"\",
    \"dns_srv\": false,
    \"host\": \"$DATABASE\",
    \"name\": \"tibit_users\",
    \"username\": \"pdmUserAdmin\",
    \"password\": \"v3nk0Admin\",
    \"port\": \"27017\",
    \"db_uri\": \"\",
    \"replica_set_enable\": false,
    \"replica_set_name\": \"rs0\",
    \"replica_set_hosts\": [
        \"$DATABASE:27017\"
    ],
    \"django_key\": \"$DJANGOKEY\",
    \"compression\":false
}
" > /etc/tibit/ponmgr/user_database.json

echo "Venko database files...."
cp logos/*.png /opt/tibit/ponmgr/ponmgr/assets/branding
cp footer/*.txt /opt/tibit/ponmgr/ponmgr/assets/branding

echo "Configuring Service Configs DB for PonManager"
rm -f /opt/tibit/ponmgr/api/databaseSeedFiles/documents/srvConfigs/*
cp databaseSeedFiles/srvConfigs/*.json /opt/tibit/ponmgr/api/databaseSeedFiles/documents/srvConfigs/
chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/documents/srvConfigs/*
cp databaseSeedFiles/firmwares/olts/* /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/olts
mkdir -p /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/onus
cp databaseSeedFiles/firmwares/onus/* /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/onus
chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/onus/*
chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/olts/*
chown www-data:www-data /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/onus/*
chown www-data:www-data  /opt/tibit/ponmgr/api/databaseSeedFiles/firmwares/olts/*
rm -f /opt/tibit/ponmgr/api/databaseSeedFiles/documents/pictures/onus/*
rm -f /opt/tibit/ponmgr/api/databaseSeedFiles/documents/pictures/olts/*
rm -f /opt/tibit/ponmgr/api/databaseSeedFiles/documents/pictures/controllers/*
cp databaseSeedFiles/pictures/olts/* /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/olts/
cp databaseSeedFiles/pictures/onus/* /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/onus/
cp databaseSeedFiles/pictures/controllers/* /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/controllers/
chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/controllers/*
chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/onus/*
chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/olts/*
chown www-data:www-data  /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/controllers/*
chown www-data:www-data /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/onus/*
chown www-data:www-data /opt/tibit/ponmgr/api/databaseSeedFiles/pictures/olts/*
cp metadata.json /opt/tibit/ponmgr/api/databaseSeedFiles/
chmod 644 /opt/tibit/ponmgr/api/databaseSeedFiles/metadata.json

setFirmwareDownloadParams
setAlarmDefaultConfig

echo "Starting MCMS PON Manager"
systemctl restart apache2.service
sleep 5

checkApacheRunningOK
}


configNetconf()
{
echo "Creating configuration files"
echo "Venko Netconf Scripts"
mkdir -p  /opt/tibit/netconf/examples/venko
cp -R $INITDIR/$NETCONFVER-Netconf-ConfD-$PLATFORM/examples/venko/* /opt/tibit/netconf/examples/venko


systemctl start tibit-netconf.service
sleep 5

# Installation of MCMS Netconf server

  echo "
  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

    Installing MCMS NETCONF ConfD Server $NETCONFVER

  * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
  
  $INITDIR/$NETCONFVER-Netconf-ConfD-$PLATFORM/
  Executing: apt-get install ./$NetconfImage -y
  "
  cd $INITDIR/$NETCONFVER-Netconf-ConfD-$PLATFORM/
  apt-get install ./$NetconfImage -y
  cd $INITDIR
  sleep 5

  sed -i '/\"host\":/ c\        \"host\": \"'$DATABASE'\",' /etc/tibit/netconf/NetconfInit.json 
  sed -i '/\"name\":/ c\        \"name\": \"'$NAME'\",' /etc/tibit/netconf/NetconfInit.json 
  sed -i '/\"auth_enable\":/ c\        \"auth_enable\": true,' /etc/tibit/netconf/NetconfInit.json 
  sed -i '/\"username\":/ c\        \"username\": \"pdmNetconf\",' /etc/tibit/netconf/NetconfInit.json 
  sed -i '/\"password\":/ c\        \"password\": \"V3nk0Net\",' /etc/tibit/netconf/NetconfInit.json 

## VENKO Customization: install python3-ncclient
apt-get install -y python3-ncclient
## END VENKO Customization: install python3-ncclient

#Check NETCONF Install status
STATUS="$(systemctl is-active tibit-netconf.service)"
if [[ ! "${STATUS}" = "active" ]]; then
    systemctl status --no-pager --lines 0 tibit-netconf.service
    beep
    echo "

    * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

      ERROR: NETCONF Service was not installed properly

      Exiting install script

    * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

    "
    exit 2
fi
  # Enabling user for access to Netconf server
  LOGIN_NAME="$(logname)"
  /opt/tibit/netconf/examples/nacm/nacm-config.sh /opt/tibit/netconf/examples/nacm/nacm-example.xml
  /opt/tibit/netconf/examples/nacm/nacm-config.sh netconf-user.xml
  /opt/tibit/netconf/examples/nacm/nacm-add-user.sh ${LOGIN_NAME} admin

  systemctl restart tibit-netconf.service

  sleep 5
  echo "NETCONF Server install complete!"
  systemctl status --no-pager --lines 0 tibit-netconf.service

}

configPonController()
{

echo "Removing old SRV"
rm -f /opt/tibit/poncntl/database/SRV*.json

if [[ "$VLAN" =  "4090" ]]; then
  echo "Activating venko-vlan_setup.service"
  echo "#!/bin/bash

ip link set ${INTER} up
ip link add link ${INTER} name ${INTER}.s4090 type vlan id 4090 egress-qos-map 0:7 1:7 2:7 3:7 4:7 5:7 6:7
ip link set dev ${INTER}.s4090 up
" > /opt/tibit/setup-vlan.sh
  chmod +x /opt/tibit/setup-vlan.sh
  /opt/tibit/setup-vlan.sh
  echo "[Unit]
Description=Venko Create vlan interface for Poncntl
After=network-online.target
Wants=network-online.target
Before=tibit-poncntl.service

[Service]
ExecStart=/opt/tibit/setup-vlan.sh
Type=exec
Restart=no

[Install]
WantedBy=user.target
RequiredBy=tibit-poncntl.service
" > /etc/systemd/system/venko-vlan_setup.service
  sudo systemctl daemon-reload
  sudo systemctl enable venko-vlan_setup.service
  sudo systemctl start venko-vlan_setup.service
  TIBITINTERFACE="$INTER.s4090"
  sleep 10
else
  echo "Configuring PONCNTL interface wihtout vlan"
  TIBITINTERFACE=$INTER
fi

echo "Creating configuration file..."
echo "
{
    \"CNTL\": {
        \"Auth\": false,
        \"CFG Version\": \"$PONCNTLVER\",
        \"DHCPv4\": true,
        \"DHCPv6\": true,
        \"PPPoE\": true,
        \"UMT interface\": \"tibitvirt\"
    },
    \"DEBUG\": {},
    \"JSON\": {
        \"databaseDir\": \"/opt/tibit/poncntl/database/\",
        \"defaultDir\": \"/opt/tibit/poncntl/database/\"
    },
    \"Local Copy\": {
        \"CNTL-STATE\": false,
        \"OLT-STATE\": false,
        \"ONU-STATE\": false
    },
    \"Logging\": {
        \"Directory\": \"/var/log/tibit\",
        \"FileCount\": 3,
        \"FileSize\": 1024000,
        \"Tracebacks\": true
    },
    \"MongoDB\": {
      \"auth_db\": \"tibit_users\",
      \"auth_enable\": true,
      \"ca_cert_path\": \"/etc/tibit/ca.pem\",
      \"host\": \"$DATABASE\",
      \"name\": \"$NAME\",
      \"password\": \"V3nk0Con\",
      \"port\": \"27017\",
      \"tls_enable\": false,
      \"username\": \"pdmPonController\",
      \"dns_srv\": false,
      \"db_uri\": \"\",
      \"replica_set_enable\": false,
      \"replica_set_name\": \"rs0\",
      \"replica_set_hosts\":
          [
          \"10.10.10.5:27017\",
          \"10.20.10.20:27999\",
          \"mongo02.example.com:17999\"
          ],
      \"valicate_cfg\": true
  },
    \"databaseType\": \"MongoDB\",
    \"interface\": \"$TIBITINTERFACE\"
}
" > /etc/tibit/poncntl/PonCntlInit.json

  if [[ "$VLAN" =  "4090" ]]; then
    checkVenkoVlanRunningOK
  fi
  echo "Starting tibit-poncntl....."
  systemctl restart tibit-poncntl.service
  sleep 2

  checkPonControllerRunningOK
}


# Check to see if root or sudo
if [[ $(id -u) -ne 0 ]] ; then
    echo "Please run as root or sudo"
fi

echo -e "*********************************************************************\n"
echo -e "Finishing the installation for  Tibit Version $VERSION \n"
echo -e "*********************************************************************\n"
echo -e "*********************************************************************\n"
echo -e "Now choose the network interface and the vlan used to connect to OLT. \n"
echo -e "*********************************************************************\n"
askForInterface
askForVlan
echo -e "*********************************************************************\n"
echo -e "Now choose the IP and PORT for MongoDB connection. \n"
echo -e "*********************************************************************\n"
askForMongoIP
echo -e "*********************************************************************\n"
echo -e "Configuring MongoDB. \n"
echo -e "*********************************************************************\n"
configMongo
echo -e "*********************************************************************\n"
echo -e "Configuring PonManager $PONMGRVER. \n"
echo -e "*********************************************************************\n"
configPonManager
echo -e "*********************************************************************\n"
echo -e "Configuring NetConf $NETCONFVER. \n"
echo -e "*********************************************************************\n"
configNetconf
echo -e "*********************************************************************\n"
echo -e "Configuring Pon Controller $PONCNTLVER. \n"
echo -e "*********************************************************************\n"
configPonController
./status.sh
FILE="/etc/tibit_preinstall"
if [ -f "$FILE" ]; then
	rm /etc/tibit_preinstall
fi
echo -e "*********************************************************************\n"
echo -e "Tibit $VERSION installed. \n"
echo -e "*********************************************************************\n"
exit 0

