#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
""" Get performance monitoring statistics for an ONU.

This MCMS REST API example script retrieves performance monitoring
statistics for an ONU. The script returns the accumulated statistics for the
last hour by default. The --hours and --days options can be used to retreive
accumulated statistics collected over longer durations.

Example:

  ./get_onu_stats.py --url https://10.2.10.29/api --user <email> --password <password> --onu ALPHe30cadcf


usage: get_onu_stats.py [-d DATABASE] [--days DAYS] [-h] [--hours HOURS]
                        [-l URL] --onu ONU [-p PASSWORD] [-u USER] [-v]

optional arguments:
  -d DATABASE, --db DATABASE
                        Name of the database. (default: Default)
  --days DAYS           Display ONU statistics collected over the last 1..7
                        days (default: 0)
  -h, --help            Show this help message and exit.
  --hours HOURS         Display ONU statistics collected over the last 1..24
                        hours (default: 1)
  -l URL, --url URL     URL of the MCMS API server (e.g.,
                        https://10.2.10.29/api). (default:
                        https://10.2.10.29/api)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  -p PASSWORD, --password PASSWORD
                        User password to authenticate with. (default: tibit)
  -u USER, --user USER  User email to authenticate with. (default:
                        tibit@tibitcom.com)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
from collections import Counter
from datetime import datetime, timedelta
import sys
from api_client import ApiClient
from api_utilities import dict_read, slot12_8ns_to_us

def print_olt_service_port_stats(stats):
    """
    Print statistics for an OLT Service Port or an ONU Managemnet Channel (OMCC).

    Args:
        stats: OLT Service Port or OMCC statistics objects
    """
    # Header definitions
    tx_str = "TX"
    rx_str = "RX"
    sp_str = ""
    print(f"                           {tx_str:>{20}} {rx_str:>{20}}")
    print(f" 64 Byte Frames:           {stats['TX Frames 64']:>{20}} {stats['RX Frames 64']:>{20}}")
    print(f" 65_127 Byte Frames:       {stats['TX Frames 65_127']:>{20}} {stats['RX Frames 65_127']:>{20}}")
    print(f" 128_255 Byte Frames:      {stats['TX Frames 128_255']:>{20}} {stats['RX Frames 128_255']:>{20}}")
    print(f" 256_511 Byte Frames:      {stats['TX Frames 256_511']:>{20}} {stats['RX Frames 256_511']:>{20}}")
    print(f" 512_1023 Byte Frames:     {stats['TX Frames 512_1023']:>{20}} {stats['RX Frames 512_1023']:>{20}}")
    print(f" 1024_1518 Byte Frames:    {stats['TX Frames 1024_1518']:>{20}} {stats['RX Frames 1024_1518']:>{20}}")
    print(f" Total Frames:             {stats['TX Frames green']:>{20}} {stats['RX Frames green']:>{20}}")
    print()
    print(f" Unicast Octets:           {stats['TX Unicast Octets']:>{20}} {stats['RX Unicast Octets']:>{20}}")
    print(f" Bcast and Mcast Octets:   {stats['TX Multi/Broadcast Octets']:>{20}} {stats['RX Multi/Broadcast Octets']:>{20}}")
    print(f" Total Octets:             {stats['TX Total Octets']:>{20}} {stats['RX Total Octets']:>{20}}")
    print()
    print(f" Encrypted Octets:         {stats['TX Encrypted Octets']:>{20}} {stats['RX Encrypted Octets']:>{20}}")
    print(f" Unencrypted Octets:       {stats['TX Plain Octets']:>{20}} {stats['RX Plain Octets']:>{20}}")
    print()
    print(f" FEC Corrected Blocks:     {sp_str:>{20}} {stats['RX FEC Corrected Blocks']:>{20}}")
    print(f" FEC Corrections:          {sp_str:>{20}} {stats['RX FEC Corrections']:>{20}}")
    print(f" FEC Good Blocks:          {sp_str:>{20}} {stats['RX FEC Good Blocks']:>{20}}")
    print(f" FEC Uncorrectable Blocks: {sp_str:>{20}} {stats['RX FEC Uncorrectable Blocks']:>{20}}")
    print()
    print(f" RX Overflow Octets:       {sp_str:>{20}} {stats['RX Overflow Octets']:>{20}}")
    print(f" RX Key Mismatch Octets:   {sp_str:>{20}} {stats['RX Key Mismatch Octets']:>{20}}")
    print(f" RX Overflow Drops:        {sp_str:>{20}} {stats['RX Overflow Drops']:>{20}}")
    print(f" RX Too Short Drops:       {sp_str:>{20}} {stats['RX Too Short Drops']:>{20}}")
    print(f" RX Too Long Drops:        {sp_str:>{20}} {stats['RX Too Long Drops']:>{20}}")
    print(f" RX Bad ICV Drops:         {sp_str:>{20}} {stats['RX Bad ICV Drops']:>{20}}")
    print(f" RX CRC8 Errorss:          {sp_str:>{20}} {stats['RX CRC8 Errors']:>{20}}")
    print(f" RX CRC32 Drops:           {sp_str:>{20}} {stats['RX CRC32 Drops']:>{20}}")
    print(f" RX HEC Errors:            {sp_str:>{20}} {stats['RX HEC Errors']:>{20}}")
    print(f" RX Errored BIP Bits:      {sp_str:>{20}} {stats['RX Errored BIP Bits']:>{20}}")
    print(f" RX Errored BIP Blocks:    {sp_str:>{20}} {stats['RX Errored BIP Blocks']:>{20}}")


def main():
    """ Entry point for the script. """
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("-d", "--db", action="store", dest="database", default="Default", required=False, help="Name of the database.")
    parser.add_argument("--days", action="store", dest="days", default=0, required=False, help="Display ONU statistics collected over the last 1..7 days")
    parser.add_argument("-h", "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("--hours", action="store", dest="hours", default=1, required=False, help="Display ONU statistics collected over the last 1..24 hours")
    parser.add_argument("-l", "--url", action="store", dest="url", default="https://10.2.10.29/api", required=False, help="URL of the MCMS API server (e.g., https://10.2.10.29/api).")
    parser.add_argument("--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-p", "--password", action="store", dest="password", default="tibit", required=False, help="User password to authenticate with.")
    parser.add_argument("-u", "--user", action="store", dest="user", default="tibit@tibitcom.com", required=False, help="User email to authenticate with.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()


    # Instantiate an API Client Connection
    api_client = ApiClient(args.url, args.verbose)

    # Login to the web server
    api_client.login(args.user, args.password)

    # Select the database to use for this session
    api_client.select_database(args.database)

    # Compute the start time for the stats time period
    if args.days:
        start_time = datetime.utcnow() - timedelta(days=int(args.days))
    else:
        start_time = datetime.utcnow() - timedelta(hours=int(args.hours))

    # Get ONU statistics
    status, onu_stats = api_client.request("GET", f"/v1/onus/stats/{args.onu}/?start-time={start_time}")
    if status != 200 or not onu_stats:
        print(f"ERROR: Failed to read statistics for ONU {args.onu}.")
        sys.exit(1)

    # Sum up OLT Service Port counters
    olt_pon_service_port_stats = Counter()
    for entry in onu_stats:
        if "OLT-PON Service 0" in entry:
            olt_pon_service_port_stats.update(entry["OLT-PON Service 0"])

    # Sum up GPON OMCC Channel counters
    olt_pon_omcc_stats = Counter()
    for entry in onu_stats:
        if "OLT-OMCC" in entry:
            olt_pon_omcc_stats.update(entry["OLT-OMCC"])

    # Get ONU state data
    status, onu_state = api_client.request("GET", f"/v1/onus/states/{args.onu}/")
    if status != 200 or not onu_state:
        print(f"ERROR: Failed to read state for ONU {args.onu}.")
        sys.exit(1)

    # Get the optical levels as reported by the OLT for this ONU
    if 'OLT-PON' in onu_state['STATS']:
        olt_optical_levels = {
            "tx": onu_state['STATS']['OLT-PON']['TX Optical Level'],
            "rx": onu_state['STATS']['OLT-PON']['RX Optical Level']
        }
    else:
        olt_optical_levels = {"tx": "-", "rx": "-"}

    # Get the optical levels as reported by the ONU
    if 'ONU-PON' in onu_state['STATS']:
        onu_optical_levels = {
            "tx": onu_state['STATS']['ONU-PON']['TX Optical Level'],
            "rx": onu_state['STATS']['ONU-PON']['RX Optical Level']
        }
    else:
        onu_optical_levels = {"tx": "-", "rx": "-"}


    # Display the statistics for the ONU
    print(f"\nStatistics for ONU {args.onu}:")
    print()

    # OMCI Channel Statistics
    print("GPON OMCI Management Channel (OMCC)")
    if olt_pon_omcc_stats:
        print_olt_service_port_stats(olt_pon_omcc_stats)
    print()

    # OLT Service Port Statistics
    print("OLT Service Port 0")
    if olt_pon_service_port_stats:
        print_olt_service_port_stats(olt_pon_service_port_stats)
    print()

    # PON Ranging Information
    print("PON Ranging:")
    print(f"  Fiber Distance:          {dict_read(onu_state, 'STATS.OLT-PON.Fiber Distance', default=0.0):>{20}} km")
    print(f"  Equalization Delay:      {slot12_8ns_to_us(dict_read(onu_state, 'STATS.OLT-PON.Equalization Delay', default=0.0)):>{20}} us")
    print(f"  One Way Delay:           {dict_read(onu_state, 'STATS.OLT-PON.One Way Delay', default=0.0):>{20}} us")
    print()

    # PON Connectivity Statistics
    print("Downstream Connection:")
    print(f"  OLT TX Power:            {olt_optical_levels['tx']:>{20}} dB")
    print(f"  ONU RX Power:            {onu_optical_levels['rx']:>{20}} dB")
    print(f"  ONU RX Pre-FEC BER:      {dict_read(onu_state, 'STATS.OLT-PON.RX Pre-FEC BER', default=0.0) / 1e12:>{20}}")
    print(f"  ONU RX Pre-FEC BER:      {dict_read(onu_state, 'STATS.OLT-PON.RX Post-FEC BER', default=0.0) / 1e12:>{20}}")
    print()
    print("Upstream Connection:")
    print(f"  OLT RX Power:            {olt_optical_levels['rx']:>{20}} dB")
    print(f"  ONU TX Power:            {onu_optical_levels['tx']:>{20}} dB")
    print(f"  OLT RX Pre-FEC BER:      {dict_read(onu_state, 'STATS.OLT-PON.RX Pre-FEC BER', default=0.0) / 1e12:>{20}}")
    print(f"  OLT RX Pre-FEC BER:      {dict_read(onu_state, 'STATS.OLT-PON.RX Post-FEC BER', default=0.0) / 1e12:>{20}}")
    print()

if __name__ == '__main__':
    main()
