#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (c) 2025, Ciena Corporation                                    #
# All rights reserved.                                                     #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
# Distributed as Ciena-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
""" Configure firmware for an ONU.

This MCMS REST API example script lists configures firmware for an ONU. The
script configures a firmware version and filename for the specified ONU
firmware bank.

Example:

  ./config_onu_firmware.py --url https://10.2.10.29/api --user <email> --password <password> \
      --onu TBITc84c0083 --bank 0 --filename "R2.2.1-ONU-FW.bin" --version "R2.2.1"


usage: config_onu_firmware.py --bank BANK [-d DATABASE] --filename FILENAME
                              [-f FILTER] [-h] [-l URL] --onu ONU
                              [-p PASSWORD] [-u USER] [-v] --version VERSION

optional arguments:
  --bank BANK           ONU firmware bank to download to (0 or 1) (default:
                        None)
  -d DATABASE, --db DATABASE
                        Name of the database. (default: Default)
  --filename FILENAME   ONU firmware file name. (default: None)
  -f FILTER, --filter FILTER
                        Filter by ONU state (e.g., Dregistered, Disabled,
                        Disallowed-Admin, Disallowed-Error, Dying-Gasp,
                        Registered, Unprovisioned, Unspecified) (default:
                        None)
  -h, --help            Show this help message and exit.
  -l URL, --url URL     URL of the MCMS API server (e.g.,
                        https://10.2.10.29/api). (default:
                        https://10.2.10.29/api)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  -p PASSWORD, --password PASSWORD
                        User password to authenticate with. (default: tibit)
  -u USER, --user USER  User email to authenticate with. (default:
                        tibit@tibitcom.com)
  -v, --verbose         Verbose output. (default: False)
  --version VERSION     ONU firmware version. (default: None)

"""

import argparse
import sys
from api_client import ApiClient
from api_utilities import dict_read

def main():
    """ Entry point for the script. """
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("--bank", action="store", dest="bank", default=None, required=True, help="ONU firmware bank to download to (0 or 1)")
    parser.add_argument("-d", "--db", action="store", dest="database", default="Default", required=False, help="Name of the database.")
    parser.add_argument("--filename", action="store", dest="filename", default=None, required=True, help="ONU firmware file name.")
    parser.add_argument("-f", "--filter", action="store", dest="filter", default=None, required=False, help="Filter by ONU state (e.g., Dregistered, Disabled, Disallowed-Admin, Disallowed-Error, Dying-Gasp, Registered, Unprovisioned, Unspecified)")
    parser.add_argument("-h", "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-l", "--url", action="store", dest="url", default="https://10.2.10.29/api", required=False, help="URL of the MCMS API server (e.g., https://10.2.10.29/api).")
    parser.add_argument("--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-p", "--password", action="store", dest="password", default="tibit", required=False, help="User password to authenticate with.")
    parser.add_argument("-u", "--user", action="store", dest="user", default="tibit@tibitcom.com", required=False, help="User email to authenticate with.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.add_argument("--version", action="store", dest="version", default=None, required=True, help="ONU firmware version.")
    parser.parse_args()
    args = parser.parse_args()


    # Instantiate an API Client Connection
    api_client = ApiClient(args.url, args.verbose)

    # Login to the web server
    api_client.login(args.user, args.password)

    # Select the database to use for this session
    api_client.select_database(args.database)

    # Validate the bank pointer value
    selected_fw_bank_ptr = int(args.bank)
    if selected_fw_bank_ptr not in (0, 1):
        print(f"ERROR: Invalid value for firmware bank {selected_fw_bank_ptr}.")
        sys.exit(1)

    onu_fw_bank_files = ["",""]
    onu_fw_bank_versions = ["",""]

    # Initialize the FW Bank Versions with values from state (i.e., start with the 'installed' versions)
    status, onu_state = api_client.request("GET", f"/v1/onus/states/{args.onu}/")
    if onu_state:
        if len(onu_state["ONU"]["FW Bank Versions"]) > 0 and onu_state["ONU"]["FW Bank Versions"][0]:
            onu_fw_bank_versions[0] = onu_state["ONU"]["FW Bank Versions"][0]
        if len(onu_state["ONU"]["FW Bank Versions"]) > 1 and onu_state["ONU"]["FW Bank Versions"][1]:
            onu_fw_bank_versions[1] = onu_state["ONU"]["FW Bank Versions"][1]

    # Get the ONU-CFG for this ONU
    status, onu_cfg = api_client.request("GET", f"/v1/onus/configs/{args.onu}/")
    if status != 200 or not onu_cfg:
        print(f"ERROR: Failed to read state for ONU {args.onu}.")
        sys.exit(1)

    # Update FW Bank Files from the existing ONU configuration
    if len(onu_cfg["ONU"]["FW Bank Files"]) > 0 and onu_cfg["ONU"]["FW Bank Files"][0]:
        onu_fw_bank_files[0] = onu_cfg["ONU"]["FW Bank Files"][0]
    if len(onu_cfg["ONU"]["FW Bank Files"]) > 1 and onu_cfg["ONU"]["FW Bank Files"][1]:
        onu_fw_bank_files[1] = onu_cfg["ONU"]["FW Bank Files"][1]

    # Update FW Bank Versions from the existing ONU configuration
    if len(onu_cfg["ONU"]["FW Bank Versions"]) > 0 and onu_cfg["ONU"]["FW Bank Versions"][0]:
        onu_fw_bank_versions[0] = onu_cfg["ONU"]["FW Bank Versions"][0]
    if len(onu_cfg["ONU"]["FW Bank Versions"]) > 1 and onu_cfg["ONU"]["FW Bank Versions"][1]:
        onu_fw_bank_versions[1] = onu_cfg["ONU"]["FW Bank Versions"][1]

    # Apply the versions specified by the script arguments
    onu_fw_bank_files[selected_fw_bank_ptr] = args.filename
    onu_fw_bank_versions[selected_fw_bank_ptr] = args.version

    # Update the firmware information for the ONU
    onu_cfg["ONU"]["FW Bank Files"] = onu_fw_bank_files
    onu_cfg["ONU"]["FW Bank Versions"] = onu_fw_bank_versions

    # Update the ONU configuration
    status, result = api_client.request("PUT", f"/v1/onus/configs/{args.onu}/", data={"data" : onu_cfg})
    if status not in (200, 201):
        print(f"ERROR: Configuration failed for ONU {args.onu}.")
        sys.exit(1)

    # Logout of the web server to terminate the session
    api_client.logout()

    # Success
    print(f"Firmware bank {selected_fw_bank_ptr} updated with filename {args.filename}, version {args.version} for ONU {args.onu}")

if __name__ == '__main__':
    main()
