#!/usr/bin/env python3
#--------------------------------------------------------------------------#
#  Copyright (c) 2020-2024, Ciena Corporation                              #
#  All rights reserved.                                                    #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
#  PROPRIETARY NOTICE                                                      #
#  This Software consists of confidential information.                     #
#  Trade secret law and copyright law protect this Software.               #
#  The above notice of copyright on this Software does not indicate        #
#  any actual or intended publication of such Software.                    #
#                                                                          #
#--------------------------------------------------------------------------#

""" Get statistics for an ONU.

This Tibit YANG Example script retrieves statistics for an ONU. This example provides the option
of retrieving historic ONU statistics for the last 15-minute time slot or
retrieving the current ONU statistics being accumulated for the next 15-minute
slot.

Example:

  ./get_onu_stats.py --onu ALPHe30cadcf


usage: get_onu_stats.py [--help] [-h HOST] --onu ONU [-w PASSWD] [-p PORT]
                        [-u USER] [-v]

optional arguments:
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  -w PASSWD, --passwd PASSWD
                        Password. If no password is provided, the user will be
                        prompted to enter. (default: None)
  -p PORT, --port PORT  NETCONF Server port number. (default: 830)
  -u USER, --user USER  Username. (default: None)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
import sys
from lxml import etree
from netconf_driver import NetconfDriver

def print_olt_service_port_stats(stats):
    """
    Print statistics for an OLT Service Port or an ONU Management Channel (OMCC).

    Args:
        stats: OLT Service Port or OMCC statistics objects
    """
    # Header definitions
    tx_str = "TX"
    rx_str = "RX"
    sp_str = ""
    print(f"                           {tx_str:>{20}s} {rx_str:>{20}s}")
    print(f" 64 Byte Frames:           {stats.get('tx-frames-64', '-'):>{20}s} {stats.get('rx-frames-64', '-'):>{20}s}")
    print(f" 65_127 Byte Frames:       {stats.get('tx-frames-65_127', '-'):>{20}s} {stats.get('rx-frames-65_127', '-'):>{20}s}")
    print(f" 128_255 Byte Frames:      {stats.get('tx-frames-128_255', '-'):>{20}s} {stats.get('rx-frames-128_255', '-'):>{20}s}")
    print(f" 256_511 Byte Frames:      {stats.get('tx-frames-256_511', '-'):>{20}s} {stats.get('rx-frames-256_511', '-'):>{20}s}")
    print(f" 512_1023 Byte Frames:     {stats.get('tx-frames-512_1023', '-'):>{20}s} {stats.get('rx-frames-512_1023', '-'):>{20}s}")
    print(f" 1024_1518 Byte Frames:    {stats.get('tx-frames-1024_1518', '-'):>{20}s} {stats.get('rx-frames-1024_1518', '-'):>{20}s}")
    print(f" Total Frames:             {stats.get('tx-frames-green', '-'):>{20}s} {stats.get('rx-frames-green', '-'):>{20}s}")
    print()
    print(f" Unicast Octets:           {stats.get('tx-unicast-octets', '-'):>{20}s} {stats.get('rx-unicast-octets', '-'):>{20}s}")
    print(f" Bcast and Mcast Octets:   {stats.get('tx-multi-broadcast-octets', '-'):>{20}s} {stats.get('rx-multi-broadcast-octets', '-'):>{20}s}")
    print(f" Total Octets:             {stats.get('tx-total-octets', '-'):>{20}s} {stats.get('rx-total-octets', '-'):>{20}s}")
    print()
    print(f" Encrypted Octets:         {stats.get('tx-encrypted-octets', '-'):>{20}s} {stats.get('rx-encrypted-octets', '-'):>{20}s}")
    print(f" Unencrypted Octets:       {stats.get('tx-plain-octets', '-'):>{20}s} {stats.get('rx-plain-octets', '-'):>{20}s}")
    print()
    print(f" FEC Corrected Blocks:     {sp_str:>{20}s} {stats.get('rx-fec-corrected-blocks', '-'):>{20}s}")
    print(f" FEC Corrections:          {sp_str:>{20}s} {stats.get('rx-fec-corrections', '-'):>{20}s}")
    print(f" FEC Good Blocks:          {sp_str:>{20}s} {stats.get('rx-fec-good-blocks', '-'):>{20}s}")
    print(f" FEC Uncorrectable Blocks: {sp_str:>{20}s} {stats.get('rx-fec-uncorrectable-blocks', '-'):>{20}s}")

if __name__ == '__main__':
    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument(      "--history", action="store_true", dest="history", default=False, required=False, help="Print ONU statistics for the last 15-minute time slot.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, the user will be prompted to enter.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Build an options dictionary from the command line arguments
    options = {
        "{{ONU}}": args.onu
    }

    stats_root = None
    NSMAP = {
        'nc': "urn:ietf:params:xml:ns:netconf:base:1.0",
        'tibitcntlr': "urn:com:tibitcom:ns:yang:controller:db",
        }

    if args.history:
        # Send a Netconf <get> request to retreive the ONU statistics history (last 15-minute slot)
        ONU_STATISTICS = '''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db"
            message-id="34566760">
            <get>
                <filter type="subtree">
                    <tibitcntlr:pon-state xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db">
                        <tibitcntlr:onu-stats>
                            <tibitcntlr:onu>
                                <name>{{ONU}}</name>
                            </tibitcntlr:onu>
                        </tibitcntlr:onu-stats>
                    </tibitcntlr:pon-state>
                </filter>
            </get>
        </rpc>
        '''
        rsp_xml = nc.get(data_xml=ONU_STATISTICS, options=options, message="/tibit-pon-controller-db::pon-state/tibit-pon-controller-db:onu-stats/tibit-pon-controller-db:onu/tibit-pon-controller-db:statistics")
        if rsp_xml:
            root = etree.fromstring(rsp_xml)
            if root is not None:
                stats_root = root.find("nc:data/tibitcntlr:pon-state/tibitcntlr:onu-stats/tibitcntlr:onu/tibitcntlr:statistics", namespaces=NSMAP)
    else:
        # Send a Netconf <get> request to retreive the current ONU statistics (being accumulated for the next 15-minute slot)
        ONU_STATISTICS = '''
        <rpc
            xmlns="urn:ietf:params:xml:ns:netconf:base:1.0"
            xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db"
            message-id="34566760">
            <get>
                <filter type="subtree">
                    <tibitcntlr:pon-state xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db">
                        <tibitcntlr:onu-state>
                            <tibitcntlr:onu>
                                <name>{{ONU}}</name>
                                <tibitcntlr:statistics/>
                            </tibitcntlr:onu>
                        </tibitcntlr:onu-state>
                    </tibitcntlr:pon-state>
                </filter>
            </get>
        </rpc>
        '''
        rsp_xml = nc.get(data_xml=ONU_STATISTICS, options=options, message="/tibit-pon-controller-db::pon-state/tibit-pon-controller-db:onu-state/tibit-pon-controller-db:onu/tibit-pon-controller-db:statistics")
        if rsp_xml:
            root = etree.fromstring(rsp_xml)
            if root is not None:
                stats_root = root.find("nc:data/tibitcntlr:pon-state/tibitcntlr:onu-state/tibitcntlr:onu/tibitcntlr:statistics", namespaces=NSMAP)

    # Parse the Netconf response and retrieve the OLT PON statistics from the XML response data.
    olt_pon_stats = {}
    if stats_root is not None:
        stats = stats_root.findall("tibitcntlr:olt-pon/*", namespaces=NSMAP)
        if stats is not None:
            for stat in stats:
                # Strip the XML namespace from the tag
                tag = stat.tag.split("}")[1][0:]
                olt_pon_stats[tag] = stat.text

    # Parse the Netconf response and retrieve the OLT Service Port statistics from the XML response data.
    olt_pon_service_port_stats = {}
    if stats_root is not None:
        stats = stats_root.findall("tibitcntlr:olt-pon-service/*", namespaces=NSMAP)
        if stats is not None:
            for stat in stats:
                # Strip the XML namespace from the tag
                tag = stat.tag.split("}")[1][0:]
                olt_pon_service_port_stats[tag] = stat.text

    # Parse the Netconf response and retrieve the OMCI Management Channel (OMCC) statistics from the XML response data.
    olt_pon_omcc_stats = {}
    if stats_root is not None:
        stats = stats_root.findall("tibitcntlr:olt-pon-omcc/*", namespaces=NSMAP)
        if stats is not None:
            for stat in stats:
                # Strip the XML namespace from the tag
                tag = stat.tag.split("}")[1][0:]
                olt_pon_omcc_stats[tag] = stat.text

    # Parse the Netconf response and retrieve the ONU PON statistics from the XML response data.
    onu_pon_stats = {}
    if stats_root is not None:
        stats = stats_root.findall("tibitcntlr:onu-pon/*", namespaces=NSMAP)
        if stats is not None:
            for stat in stats:
                # Strip the XML namespace from the tag
                tag = stat.tag.split("}")[1][0:]
                onu_pon_stats[tag] = stat.text

    # Display the statistics for the ONU
    print(f"\nStatistics for ONU {args.onu}:")
    # OMCI Channel Statistics
    print("GPON OMCI Management Channel (OMCC)")
    if olt_pon_omcc_stats:
        print_olt_service_port_stats(olt_pon_omcc_stats)
    print()
    # OLT Service Port Statistics
    print("OLT Service Port 0")
    if olt_pon_service_port_stats:
        print_olt_service_port_stats(olt_pon_service_port_stats)
    print()
    # Optical Levels and BER
    print("Optical Levels")
    print(" Downstream")
    if olt_pon_stats and onu_pon_stats:
        print(f"  OLT TX:              {olt_pon_stats.get('tx-optical-level', '-'):>{7}s} dB")
        print(f"  ONU RX:              {onu_pon_stats.get('rx-optical-level', '-'):>{7}s} dB")
    print(" Upstream")
    if olt_pon_stats and onu_pon_stats:
        print(f"  OLT RX:              {olt_pon_stats.get('rx-optical-level', '-'):>{7}s} dB")
        print(f"  ONU TX:              {onu_pon_stats.get('tx-optical-level', '-'):>{7}s} dB")
        if 'rx-pre-fec-ber' in olt_pon_stats:
            print(f"  OLT RX Pre-FEC BER:  {olt_pon_stats.get('rx-pre-fec-ber', '-'):>{7}s}")
            print(f"  OLT RX Post-FEC BER: {olt_pon_stats.get('rx-pre-fec-ber', '-'):>{7}s}")