#!/usr/bin/env python3
#--------------------------------------------------------------------------#
#  Copyright (c) 2021-2024, Ciena Corporation                              #
#  All rights reserved.                                                    #
#                                                                          #
#     _______ _____ __    __ ___                                           #
#    / _ __(_) ___//  |  / // _ |                                          #
#   / /   / / /__ / /|| / // / ||                                          #
#  / /___/ / /__ / / ||/ // /__||                                          #
# /_____/_/_____/_/  |__//_/   ||                                          #
#                                                                          #
#  PROPRIETARY NOTICE                                                      #
#  This Software consists of confidential information.                     #
#  Trade secret law and copyright law protect this Software.               #
#  The above notice of copyright on this Software does not indicate        #
#  any actual or intended publication of such Software.                    #
#                                                                          #
#--------------------------------------------------------------------------#

""" Configure Firmware Upgrade for an ONU.

This Tibit YANG Example script configures firmware upgrade for an ONU. The script
provides an example for disabling firmware upgrade for an ONU.

Example - Disable firmware upgrade for the ONU:

  ./firmware_upgrade/edit_config_disable_firmware_upgrade.py --onu ALPHe30cadcf


usage: edit_config_onu_firmware.py [--activate-bank ACTIVATE_BANK] [--help]
                                   [-h HOST] --onu ONU [-w PASSWD] [-p PORT]
                                   [-u USER] [-v]

optional arguments:
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  -w PASSWD, --passwd PASSWD
                        Password. If no password is provided, the user will be
                        prompted to enter. (default: None)
  -p PORT, --port PORT  NETCONF Server port number. (default: 830)
  -u USER, --user USER  Username. (default: None)
  -v, --verbose         Verbose output. (default: False)

"""

import argparse
from lxml import etree
import os
import sys
sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
from netconf_driver import NetconfDriver

if __name__ == '__main__':
    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-w", "--passwd", action="store", dest="passwd", default=None, required=False, help="Password. If no password is provided, the user will be prompted to enter.")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()

    nc = NetconfDriver(host=args.host, port=args.port, user=args.user, passwd=args.passwd, verbose=args.verbose)
    if not nc:
        # Error
        print(f"ERROR: Failed to connect to Netconf server {args.host}:{args.port}.")
        sys.exit(1)

    # Build an options dictionary from the command line arguments
    # Setting the Firmware bank pointer to 65535 and clearing the firmware image
    # configuration disables firmware download on this ONU.
    options = {
        "{{ONU}}" : args.onu,
        "{{ACTIVE_BANK}}" : 65535,
        "{{FW_FILE_BANK0}}" : "",
        "{{FW_VERSION_BANK0}}" : "",
        "{{FW_FILE_BANK1}}" : "",
        "{{FW_VERSION_BANK1}}" : "",
    }

    # Configure the firmware image for a bank
    nc.edit_config(filename="1-onu-cfg-firmware-upgrade.xml", options=options)

    # Display a summary of what was configured
    print(f"\nDisabled Firmware Upgrade for ONU {options['{{ONU}}']}")
    print(f" Firmware Bank Pointer:  {options['{{ACTIVE_BANK}}']}")
    print(f" Bank 0:")
    print(f"   Version: {options['{{FW_VERSION_BANK0}}']}")
    print(f"   File:    {options['{{FW_FILE_BANK0}}']}")
    print(f" Bank 1:")
    print(f"   Version: {options['{{FW_VERSION_BANK1}}']}")
    print(f"   File:    {options['{{FW_VERSION_BANK1}}']}")