#!/usr/bin/env bash
# --------------------------------------------------------------------------#
#  Copyright (C) 2021 by Tibit Communications, Inc.                         #
#  All rights reserved.                                                     #
#                                                                           #
#     _______ ____  _ ______                                                #
#    /_  __(_) __ )(_)_  __/                                                #
#     / / / / __  / / / /                                                   #
#    / / / / /_/ / / / /                                                    #
#   /_/ /_/_____/_/ /_/                                                     #
#                                                                           #
#                                                                           #
#  Distributed as Tibit-Customer confidential                               #
#                                                                           #
# --------------------------------------------------------------------------#

TMPFILE=$(mktemp /tmp/tibit_nc_rpc.XXXXXX)

# Usage
usage () {
    cat <<HELP_USAGE

    $0 onu-name 
    or
    $0

    List optical level of ONUs configured on the NETCONF server.

    Options:
      -h|--help       Display this help and usage text.
      -p|--port       TCP port number used to connect to the NETCONF Server (default: 830).
      -s|--server     Hostname or IP address used to connect to the NETCONF Server (default: localhost).
      -u|--user       User used to connect to the NETCONF Server

    Example:
      List the Optical Level for a specific ONU
      $0 GPON20900039
      List Optical Level for all onus of system
      $0
HELP_USAGE

exit 0
}

# Defaults
HOST=localhost
PORT=830

# Parse command line arguments
PARAMS=""
while (( "$#" )); do
    case "$1" in
        -s|--server)
            HOST=$2
            shift 2
            ;;
        -p|--port)
            PORT=$2
            shift 2
            ;;
        -u|--user)
            USER=$2
            shift 2
            ;;
        -h|--help)
            usage
            ;;
        --) # end argument parsing
            shift
            break
            ;;
        -*|--*=) # unsupported option
            echo "Error: Unsupported option $1" >&2
            usage
            ;;
        *) # preserve positional arguments
            PARAMS="$PARAMS $1"
            shift
            ;;
    esac
done
# set positional arguments in their proper place
eval set -- "$PARAMS"
if [ -z "$1" ]
then
   NAME="<tibitcntlr:name/>"
else
   NAME="<tibitcntlr:name>"$1"</tibitcntlr:name>"
fi

if [ -z "$USER" ]
then
   LOGIN=""
else
   LOGIN="--login $USER"
fi

cat >${TMPFILE}  << EDITCONFIGXML
<get xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
    <filter type="subtree">
        <tibitcntlr:onu-state xmlns:tibitcntlr="urn:com:tibitcom:ns:yang:controller:db">
         <tibitcntlr:onu>
            $NAME
             <tibitcntlr:statistics>
                   <tibitcntlr:olt-pon>
                         <tibitcntlr:rx-optical-level/>
                         <tibitcntlr:tx-optical-level/>
                   </tibitcntlr:olt-pon>
                   <tibitcntlr:onu-pon>
                         <tibitcntlr:rx-optical-level/>
                         <tibitcntlr:tx-optical-level/>
                   </tibitcntlr:onu-pon>
             </tibitcntlr:statistics>
           </tibitcntlr:onu>
        </tibitcntlr:onu-state>
    </filter>
</get>
EDITCONFIGXML

PATH=${PATH}:/opt/tibit/netconf/bin netopeer2-cli <<COMMANDINPUT
connect --host ${HOST} --port ${PORT} ${LOGIN}
user-rpc --content ${TMPFILE} --rpc-timeout 120
exit
COMMANDINPUT

rm -rf ${TMPFILE}
