#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2021 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#--------------------------------------------------------------------------#

""" Remove an ONU from the OLT Inventory..

This Tibit YANG script will remove from OLT Inventory the ONU.
The UNO Configuration will not be removed from the system, it will just be an unauthorized ONU  or not inventoried in the OLT.

Example - Remove ONU ALPHe30cadcf from OLT 70:b3:d5:52:37:24.

  ./edit_config_unprovision_onu.py \
      --olt 70:b3:d5:52:37:24 \
      --onu ALPHe30cadcf \

usage: edit_config_unprovision_onu.py [--help] [-h HOST] [-p PORT]
                                   --olt OLT
                                   --onu ONU 

optional arguments:
  --help                Show this help message and exit.
  -h HOST, --host HOST  NETCONF Server IP address or hostname. (default:
                        127.0.0.1)
  --olt OLT             OLT MAC Address (e.g., 70:b3:d5:52:37:24) (default:
                        None)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
"""

import argparse
import itertools
from lxml import etree
import os
import sys
import getpass

sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), ".."))
from common.netconf_driver import NetconfDriver

if __name__ == '__main__':
    # Command line arguments
    parser = argparse.ArgumentParser(add_help=False,formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument(      "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-h", "--host", action="store", dest="host", default='127.0.0.1', required=False, help="NETCONF Server IP address or hostname.")
    parser.add_argument(      "--olt", action="store", dest="olt", default=None, required=True, help="OLT MAC Address (e.g., 70:b3:d5:52:37:24)")
    parser.add_argument(      "--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-p", "--port", action="store", dest="port", default='830', required=False, help="NETCONF Server port number.")
    parser.add_argument("-u", "--user", action="store", dest="user", default=None, required=False, help="Username.")
    parser.parse_args()
    args = parser.parse_args()

    if args.user:
        user = args.user
    else:
        user = getpass.getuser()

    passwd = getpass.getpass(prompt="Please enter the password for "+user+": ")

    nc = NetconfDriver(host=args.host, port=args.port, user=user, passwd=passwd, verbose=False)
    if nc:

        options = {
            "{{OLT}}" : args.olt,
            "{{ONU}}" : args.onu,
        }


        data_xml = ''' <rpc xmlns="urn:ietf:params:xml:ns:netconf:base:1.0" message-id="34566760"> 
                     <edit-config>
                       <target>
                         <running />
                       </target>
                       <config>
                          <olt xmlns=\"urn:com:tibitcom:ns:yang:controller:db\">
                              <olt>
                                 <name>{{OLT}}</name>
                                 <device-id>{{OLT}}</device-id>
                                 <!-- ONU Inventory -->
                                 <onus xmlns:nc="urn:ietf:params:xml:ns:netconf:base:1.0" nc:operation="remove">
                                    <id>{{ONU}}</id>
                                 </onus>
                               </olt>
                           </olt>
                       </config>
                     </edit-config>
                  </rpc>'''

        # Configure the ONU ID and Alloc ID in the ONU inventory
        nc.edit_config(data_xml=data_xml,options=options,message="Removing an ONU at inventory")
        # Display a summary of what was configured
        print(f"\nUnprovisioned ONU {options['{{ONU}}']} to OLT {options['{{OLT}}']}")
