"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import pymongo.errors

from django.contrib.auth.decorators import permission_required
from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.exceptions import APIException
from rest_framework.fields import JSONField
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, OpenApiParameter, inline_serializer
from drf_spectacular.types import OpenApiTypes
from rest_framework import status

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.tools import get_nested_value, PonManagerApiResponse, permission_required_any_of


# ==================================================
# ========= One PON Automation State View ==========
# ==================================================
class OneState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_pon_automation_state",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['automation', 'state', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_automation', raise_exception=True))
    def get(self, request, pon_auto_id, version):
        """Get the state for the specified PON Automation"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="AUTO-STATE", query={"_id": pon_auto_id})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="delete_one_pon_automation_state",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['automation', 'state', 'delete']
    )
    @method_decorator(permission_required('global_config.can_delete_automation', raise_exception=True))
    def delete(self, request, pon_auto_id, version):
        """Delete the state of the specified PON Automation"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="AUTO-STATE", query={"_id": pon_auto_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========== PON Automation States View ============
# ==================================================
class States(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_pon_automation_states",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['automation', 'state', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_automation', raise_exception=True))
    def get(self, request, version):
        """Get the states for all PON Automation services"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="AUTO-STATE")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ===== One PON Automation Configuration View ======
# ==================================================
class OneConfiguration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_pon_automation_config",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['automation', 'config', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_automation', raise_exception=True))
    def get(self, request, pon_auto_id, version):
        """Get the config for the specified PON Automation"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="AUTO-CFG", query={"_id": pon_auto_id})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="put_one_pon_automation_config",
        request=inline_serializer(name="AUTO-CFG", fields={"data": JSONField(help_text="AUTO-CFG")}),
        responses={
            200: ResponseExample(200),
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['automation', 'config', 'put']
    )
    @method_decorator(permission_required_any_of(['global_config.can_update_automation', 'global_config.can_create_automation'], raise_exception=True))
    def put(self, request, pon_auto_id, version):
        """Update the config for the specified PON Automation"""
        data = get_nested_value(request.data, ["data"])
        data['AUTO']['CFG Change Count'] += 1
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'), collection="AUTO-CFG", query={"_id": pon_auto_id}, new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK
        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_one_pon_automation_config",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['automation', 'config', 'delete']
    )
    @method_decorator(permission_required('global_config.can_delete_automation', raise_exception=True))
    def delete(self, request, pon_auto_id, version):
        """Delete the config of the specified PON Automation"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="AUTO-CFG", query={"_id": pon_auto_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ====== PON Automation Configurations View ========
# ==================================================
class Configurations(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_pon_automation_configs",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['automation', 'config', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_automation', raise_exception=True))
    def get(self, request, version):
        """Get the configs for all PON Automation services"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="AUTO-CFG")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_pon_automation_config",
        request=inline_serializer(name="AUTO-CFG", fields={"data": JSONField(help_text="AUTO-CFG")}),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500)
        },
        tags=['automation', 'config', 'post']
    )
    @method_decorator(permission_required('global_config.can_create_automation', raise_exception=True))
    def post(self, request, version):
        """Create the provided PON Automation config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="AUTO-CFG", document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            pon_auto_id = get_nested_value(request.data, ["data", "_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details=f"PON Automation configuration with id {pon_auto_id} already exists")

        return response


# ==================================================
# ======= One PON Automation Statistics View =======
# ==================================================
class Statistics(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_pon_automation_stats",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            200: ResponseExample(200),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['automation', 'stats', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_automation', raise_exception=True))
    def get(self, request, pon_auto_id, version):
        """Get the statistics of the specified PON Automation between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, data="Parameter 'start-time' is required")
        else:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("STATS-AUTO-{}".format(pon_auto_id))
            try:
                if end_time is None:
                    res_data = list(collection.find({"Time": {"$gte": start_time}}).limit(10000))
                else:
                    res_data = list(collection.find({"Time": {"$gte": start_time, "$lte": end_time}}).limit(10000))
            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            # Add PON Automation ID to response format for easier handling in UI
            for block in res_data:
                block['mac_address'] = pon_auto_id

            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

        return response

    @extend_schema(
        operation_id="delete_one_pon_automation_stats",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['automation', 'stats', 'delete']
    )
    @method_decorator(permission_required('global_config.can_delete_automation', raise_exception=True))
    def delete(self, request, pon_auto_id, version):
        """Delete the Statistics of the specified PON Automation"""
        database = database_manager.get_database(request.session.get('database'))
        collection = database.get_collection("STATS-AUTO-{}".format(pon_auto_id))
        collection.drop()

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========= One PON Automation Logs View ===========
# ==================================================
class Logs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_pon_automation_logs",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            200: ResponseExample(200),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['automation', 'logs', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_automation', raise_exception=True))
    def get(self, request, pon_auto_id, version):
        """Get the logs of the specified PON Automation between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, data="Parameter 'start-time' is required")
        else:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("SYSLOG-AUTO-{}".format(pon_auto_id))
            try:
                if end_time is None:
                    res_data = list(collection.find({"time": {"$gte": start_time}},
                                                    {"_id": 0, "device ID": 0}))
                else:
                    res_data = list(collection.find({"time": {"$gte": start_time, "$lte": end_time}},
                                                    {"_id": 0, "device ID": 0}))
            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

        return response

    @extend_schema(
        operation_id="delete_one_pon_automation_logs",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['automation', 'logs', 'delete']
    )
    @method_decorator(permission_required('global_config.can_delete_automation', raise_exception=True))
    def delete(self, request, pon_auto_id, version):
        """Delete the Logs of the specified PON Automation"""
        database = database_manager.get_database(request.session.get('database'))
        collection = database.get_collection("SYSLOG-AUTO-{}".format(pon_auto_id))
        collection.drop()

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
