"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import copy
import pymongo.errors

from django.contrib.auth.decorators import permission_required
from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.exceptions import APIException
from rest_framework.fields import JSONField, ChoiceField, BooleanField
from drf_spectacular.utils import extend_schema, OpenApiParameter, inline_serializer
from drf_spectacular.types import OpenApiTypes
from rest_framework import status
from rest_framework.generics import GenericAPIView

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.serializers import RequestSerializer
from utils.tools import get_nested_value, PonManagerApiResponse, validate_query_params, validate_data, \
    permission_required_any_of, load_mongo_query_parameter


# ==================================================
# ============== One OLT State View ================
# ==================================================
class OneState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_olt_state",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'state', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the state for the specified OLT"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-STATE",
                                             query={"_id": olt_id})
        # Cannot add to projection here (after R2.3.0) because there cannot be a mix of 0's and 1's in the mongo projection except for "_id"
        if res_data and "_internal" in res_data:
            del res_data["_internal"]

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="delete_one_olt_state",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'state', 'delete']
    )
    @method_decorator(permission_required('network.can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the state of the specified OLT"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="OLT-STATE",
                                    query={"_id": olt_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# =============== OLT States View ==================
# ==================================================
class States(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_states",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'state', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    @validate_query_params(collection="OLT-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the states for all OLTs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="OLT-STATE",
                                                 query=query, distinct=distinct)
        # Filter out '_internal' field
        else:
            if projection:
                keys = list(projection.keys())
                if '_id' not in keys:
                    if any(projection[key] == 0 for key in keys):
                        projection['_internal'] = 0
                else:
                    keys.remove('_id')
                    if len(keys) == 0 and projection['_id'] == 0:
                        projection['_internal'] = 0
                    elif any(projection[key] == 0 for key in keys):
                        projection['_internal'] = 0
            else:
                projection = {'_internal': 0}

            res_data = database_manager.find(database_id=request.session.get('database'), collection="OLT-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ========== One OLT Configuration View ============
# ==================================================
class OneConfiguration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_olt_config",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'config', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the config for the specified OLT"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-CFG",
                                             query={"_id": olt_id})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="put_one_olt_config",
        request=inline_serializer(name="OLT-CFG", fields={"data": JSONField(help_text="OLT-CFG")}),
        responses={
            200: ResponseExample(200),
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'config', 'put']
    )
    @method_decorator(permission_required_any_of(['network.can_update_network_olts', 'network.can_create_network_olts'],
                                                 raise_exception=True))
    @validate_data(collection="OLT-CFG", resource_id_param="olt_id")
    def put(self, request, data, olt_id, version):
        """Update the config for the specified OLT"""
        data['OLT']['CFG Change Count'] += 1
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                             collection="OLT-CFG", query={"_id": olt_id},
                                                             new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK
        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_one_olt_config",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'config', 'delete']
    )
    @method_decorator(permission_required('network.can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the config of the specified OLT"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="OLT-CFG",
                                    query={"_id": olt_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# =========== OLT Configurations View ==============
# ==================================================
class Configurations(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_configs",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'config', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    @validate_query_params(collection="OLT-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get the configs for all OLTs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'), collection="OLT-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="OLT-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)
        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_olt_config",
        request=inline_serializer(name="OLT-CFG", fields={"data": JSONField(help_text="OLT-CFG")}),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500)
        },
        tags=['olt', 'config', 'post']
    )
    @method_decorator(permission_required('network.can_create_network_olts', raise_exception=True))
    @validate_data(collection="OLT-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided OLT config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="OLT-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            olt_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT,
                                             details=f"OLT configuration with id {olt_id} already exists")

        return response


# ==================================================
# ================ OLT Debug View ==================
# ==================================================
class Debug(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_debug",
        parameters=[
            OpenApiParameter(name="limit", description="Maximum Number of Debug Dumps to Return",
                             type=OpenApiTypes.NUMBER),
            OpenApiParameter(name="projection", description="Fields to Return.", type=OpenApiTypes.STR),
            OpenApiParameter(name="sort", description="Fields to sort in descending or ascending.",
                             type=OpenApiTypes.STR),
            OpenApiParameter(name="skip", description="Number of items to be skipped at beginning of response",
                             type=OpenApiTypes.NUMBER),
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'debug', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the debug data for an OLT"""
        # Can't use validate_query_params and other decorators because using a query within the path
        # Must manually put query params into dictionaries

        start_time = request.GET.get('time-start', None)
        end_time = request.GET.get('time-end', None)
        limit = int(request.GET.get('limit', 0))
        skip = int(request.GET.get('skip', 0))
        projection = load_mongo_query_parameter(request.GET.get('projection'))
        sort_dict = load_mongo_query_parameter(request.GET.get('sort'))

        query = {"device ID": olt_id, "valid": True}
        if start_time is not None and end_time is None:
            query["Time"] = {"$gte": start_time}
        elif end_time is not None and start_time is None:
            query["Time"] = {"$lte": end_time}
        elif start_time is not None and end_time is not None:
            temp_dict = {"$lte": end_time, "$gte": start_time}
            query["Time"] = temp_dict

        try:
            sort = []
            if sort_dict:
                for key, value in sort_dict.items():
                    value = int(value)
                    sort_dict[key] = value
                    sort.append((key, value))

            res_data = database_manager.find(database_id=request.session.get('database'), collection="DEBUG-OLT",
                                             query=query, sort=sort, limit=limit,
                                             skip=skip, projection=projection)
            return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        except:
            return PonManagerApiResponse(status=status.HTTP_500_INTERNAL_SERVER_ERROR)

    @extend_schema(
        operation_id="delete_olt_debug",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'debug', 'delete']
    )
    @method_decorator(permission_required('network.can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the debug data for an OLT"""
        try:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("DEBUG-OLT")
            collection.update_many({"device ID": olt_id}, {"$set": {"valid": False}})

        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)

# ==================================================
# ======= One OLT Alarm Configuration View =========
# ==================================================
class OneAlarmConfiguration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_olt_alarm_config",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'alarm-config', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_alarms', raise_exception=True))
    def get(self, request, cfg_id, version):
        """Get the specified OLT Alarm Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-ALARM-CFG",
                                             query={"_id": cfg_id})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="put_one_olt_alarm_config",
        request=inline_serializer(name="OLT-ALARM-CFG", fields={"data": JSONField(help_text="OLT-ALARM-CFG")}),
        responses={
            200: ResponseExample(200),
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'alarm-config', 'put']
    )
    @method_decorator(permission_required_any_of(
        ['global_config.can_update_global_config_alarms', 'global_config.can_create_global_config_alarms'],
        raise_exception=True))
    @validate_data(collection="OLT-ALARM-CFG", resource_id_param="cfg_id")
    def put(self, request, data, cfg_id, version):
        """Update the config for the specified OLT Alarm Config"""
        old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                             collection="OLT-ALARM-CFG", query={"_id": cfg_id},
                                                             new_document=data)
        if old_document is None:
            status_code = status.HTTP_201_CREATED
        else:
            status_code = status.HTTP_200_OK

        return PonManagerApiResponse(status=status_code, new_data=data, old_data=old_document)

    @extend_schema(
        operation_id="delete_one_olt_alarm_config",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'alarm-config', 'delete']
    )
    @method_decorator(permission_required('global_config.can_delete_global_config_alarms', raise_exception=True))
    def delete(self, request, cfg_id, version):
        """Delete the specified OLT Alarm Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="OLT-ALARM-CFG",
                                    query={"_id": cfg_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ======== OLT Alarm Configurations View ===========
# ==================================================
class AlarmConfigurations(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_alarm_configs",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'alarm-config', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_alarms', raise_exception=True))
    @validate_query_params(collection="OLT-ALARM-CFG")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all OLT Alarm Configs"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'),
                                                 collection="OLT-ALARM-CFG",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'), collection="OLT-ALARM-CFG",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_olt_alarm_config",
        request=inline_serializer(name="OLT-ALARM-CFG", fields={"data": JSONField(help_text="OLT-ALARM-CFG")}),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500)
        },
        tags=['olt', 'alarm-config', 'post']
    )
    @method_decorator(permission_required('global_config.can_create_global_config_alarms', raise_exception=True))
    @validate_data(collection="OLT-ALARM-CFG", resource_id_param=None)
    def post(self, request, data, version):
        """Create the provided OLT Alarm Config"""
        try:
            database_manager.insert_one(database_id=request.session.get('database'), collection="OLT-ALARM-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            doc_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT,
                                             details=f"OLT alarm configuration with id {doc_id} already exists")

        return response


# ==================================================
# ======= One OLT Alarm History State View =========
# ==================================================
class OneAlarmHistoryState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_olt_alarm_history",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'alarm-history', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the specified OLT Alarm History State"""
        res_data = database_manager.find_one(database_id=request.session.get('database'),
                                             collection="OLT-ALARM-HIST-STATE", query={"_id": olt_id})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="delete_one_olt_alarm_history",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'alarm-history', 'delete']
    )
    @method_decorator(permission_required('network.can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the specified OLT Alarm Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="OLT-ALARM-HIST-STATE",
                                    query={"_id": olt_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ======== OLT Alarm History States View ===========
# ==================================================
class AlarmHistoryStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_alarm_history",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'alarm-history', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    @validate_query_params(collection="OLT-ALARM-HIST-STATE")
    def get(self, request, query, projection, sort, limit, skip, next, distinct, version):
        """Get all OLT Alarm History States"""
        if distinct:
            res_data = database_manager.distinct(database_id=request.session.get('database'),
                                                 collection="OLT-ALARM-HIST-STATE",
                                                 query=query, distinct=distinct)
        else:
            res_data = database_manager.find(database_id=request.session.get('database'),
                                             collection="OLT-ALARM-HIST-STATE",
                                             query=query, projection=projection, sort=sort, limit=limit, skip=skip,
                                             next=next)

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ============ One OLT Statistics View =============
# ==================================================
class Statistics(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_olt_stats",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            200: ResponseExample(200),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'stats', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the statistics of the specified OLT between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Parameter 'start-time' is required"})
        else:
            database = database_manager.get_database(request.session.get('database'))
            try:
                state_data = database_manager.find_one(database_id=request.session.get('database'),
                                                       collection="OLT-STATE",
                                                       query={"_id": olt_id}, projection={"_id": 0, "CNTL.Version": 1})
                cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
                sub_three_one_version = False

                if float(cntl_version[1:4]) < 3.1:
                    sub_three_one_version = True

                if sub_three_one_version:
                    collection = database.get_collection("STATS-OLT-{}".format(olt_id.replace(":", "")))
                    if end_time is None:
                        res_data = list(collection.find({"Time": {"$gte": start_time}}).limit(10000))
                    else:
                        res_data = list(collection.find({"Time": {"$gte": start_time, "$lte": end_time}}).limit(10000))
                else:
                    # For new versions of the DB
                    collection = database.get_collection("STATS-OLT")
                    if end_time is None:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": olt_id},
                                {"valid": True},
                                {"Time": {"$gte": start_time}},
                            ]
                        }).limit(10000))
                    else:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": olt_id},
                                {"valid": True},
                                {"Time": {"$gte": start_time, "$lte": end_time}}
                            ]
                        }).limit(10000))

            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")
            # Add OLT ID to response format for easier handling in UI
            for block in res_data:
                block['mac_address'] = olt_id

            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

        return response

    @extend_schema(
        operation_id="delete_one_olt_stats",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'stats', 'delete']
    )
    @method_decorator(permission_required('network.can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the Statistics of the specified OLT"""
        database = database_manager.get_database(request.session.get('database'))

        try:
            state_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-STATE",
                                                   query={"_id": olt_id}, projection={"_id": 0, "CNTL.Version": 1})
            cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
            sub_three_one_version = False

            if float(cntl_version[1:4]) < 3.1:
                sub_three_one_version = True

            if sub_three_one_version:
                collection = database.get_collection("STATS-OLT-{}".format(olt_id.replace(":", "")))
                collection.drop()
            else:
                # For new versions of the DB
                collection = database.get_collection("STATS-OLT")
                collection.update_many({"device ID": olt_id}, {"$set": {"valid": False}})

        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ============ One OLT Logs View =============
# ==================================================
class Logs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_olt_logs",
        parameters=[
            OpenApiParameter(name="time-start", description="UTC timestamp to begin getting stats at",
                             type=OpenApiTypes.DATETIME, required=True),
            OpenApiParameter(name="time-end", description="UTC timestamp to stop getting stats at",
                             type=OpenApiTypes.DATETIME)
        ],
        responses={
            200: ResponseExample(200),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'logs', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the logs of the specified OLT between the start and end times"""
        start_time = request.GET.get('start-time', None)
        end_time = request.GET.get('end-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Parameter 'start-time' is required"})
        else:
            database = database_manager.get_database(request.session.get('database'))

            try:
                state_data = database_manager.find_one(database_id=request.session.get('database'),
                                                       collection="OLT-STATE",
                                                       query={"_id": olt_id}, projection={"_id": 0, "CNTL.Version": 1})
                cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
                sub_three_one_version = False

                if float(cntl_version[1:4]) < 3.1:
                    sub_three_one_version = True

                if sub_three_one_version:
                    collection = database.get_collection("SYSLOG-OLT-{}".format(olt_id.replace(":", "")))
                    if end_time is None:
                        res_data = list(collection.find({"time": {"$gte": start_time}},
                                                        {"_id": 0, "device ID": 0}))
                    else:
                        res_data = list(collection.find({"time": {"$gte": start_time, "$lte": end_time}},
                                                        {"_id": 0, "device ID": 0}))
                else:
                    collection = database.get_collection("SYSLOG-OLT")
                    if end_time is None:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": olt_id},
                                {"valid": True},
                                {"time": {"$gte": start_time}},
                            ]
                        }, {"_id": 0, "device ID": 0}).limit(10000))
                    else:
                        res_data = list(collection.find({
                            "$and": [
                                {"device ID": olt_id},
                                {"valid": True},
                                {"time": {"$gte": start_time, "$lte": end_time}}
                            ]
                        }, {"_id": 0, "device ID": 0}).limit(10000))

            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

        return response

    @extend_schema(
        operation_id="delete_one_olt_logs",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'logs', 'delete']
    )
    @method_decorator(permission_required('network.can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the Logs of the specified OLT"""
        database = database_manager.get_database(request.session.get('database'))

        try:
            state_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-STATE",
                                                   query={"_id": olt_id}, projection={"_id": 0, "CNTL.Version": 1})
            cntl_version = get_nested_value(state_data, ["CNTL", "Version"], "")
            sub_three_one_version = False

            if float(cntl_version[1:4]) < 3.1:
                sub_three_one_version = True

            if sub_three_one_version:
                collection = database.get_collection("SYSLOG-OLT-{}".format(olt_id.replace(":", "")))
                collection.drop()
            else:
                # For new versions of the DB
                collection = database.get_collection("SYSLOG-OLT")
                collection.update_many({"device ID": olt_id}, {"$set": {"valid": False}})

        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========= One OLT Automation State View ==========
# ==================================================
class OneAutomationState(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_olt_automation_state",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'automation', 'state', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """Get the Automation State of the specified OLT"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-AUTO-STATE",
                                             query={"_id": olt_id})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="delete_one_olt_automation_state",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'automation', 'state', 'delete']
    )
    @method_decorator(permission_required('network.can_delete_network_olts', raise_exception=True))
    def delete(self, request, olt_id, version):
        """Delete the specified OLT Automation State"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="OLT-AUTO-STATE",
                                    query={"_id": olt_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# =========== OLT Automation States View ===========
# ==================================================
class AutomationStates(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_automation_states",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'automation', 'state', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    # TODO @validate_query_params(collection="OLT-AUTO-STATE")
    def get(self, request, version):
        """Get the Automation States of all OLTs"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="OLT-AUTO-STATE")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ========= One OLT Automation Config View ==========
# ==================================================
class OneAutomationConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_olt_automation_config",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'automation', 'config', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    def get(self, request, cfg_id, version):
        """Get the specified OLT Automation Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-AUTO-CFG",
                                             query={"_id": cfg_id})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="put_one_olt_automation_config",
        request=inline_serializer(name="OLT-AUTO-CFG", fields={"data": JSONField(help_text="OLT-AUTO-CFG")}),
        responses={
            200: ResponseExample(200),
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'automation', 'config', 'put']
    )
    @method_decorator(permission_required_any_of(['network.can_update_network_olts', 'network.can_create_network_olts'],
                                                 raise_exception=True))
    # TODO @validate_data(collection="OLT-AUTO-CFG", resource_id_param="cfg_id")
    def put(self, request, cfg_id, version):
        """Update the config for the specified OLT Automation Config"""
        data = get_nested_value(request.data, ["data"])
        if data is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details="Request body must be of format '{ data: <OLT-AUTO-CFG> }'")
        else:
            data['_id'] = cfg_id
            old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                                 collection="OLT-AUTO-CFG", query={"_id": cfg_id},
                                                                 new_document=data)

            if old_document is None:
                response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=data, old_data=old_document)

        return response

    @extend_schema(
        operation_id="delete_one_olt_automation_config",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'automation', 'config', 'delete']
    )
    @method_decorator(permission_required('network.can_delete_network_olts', raise_exception=True))
    def delete(self, request, cfg_id, version):
        """Delete the specified OLT Automation Config"""
        database_manager.delete_one(database_id=request.session.get('database'), collection="OLT-AUTO-CFG",
                                    query={"_id": cfg_id})

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)


# ==================================================
# ========== OLT Automation Configs View ===========
# ==================================================
class AutomationConfigs(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_automation_configs",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'automation', 'config', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    # TODO @validate_query_params(collection="OLT-AUTO-CFG")
    def get(self, request, version):
        """Get the Automation Configs of all OLTs"""
        res_data = database_manager.find(database_id=request.session.get('database'), collection="OLT-AUTO-CFG")

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="post_olt_automation_config",
        request=inline_serializer(name="OLT-AUTO-CFG", fields={"data": JSONField(help_text="OLT-AUTO-CFG")}),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500)
        },
        tags=['olt', 'automation', 'config', 'post']
    )
    @method_decorator(permission_required('network.can_create_network_olts', raise_exception=True))
    # TODO @validate_data(collection="OLT-AUTO-CFG", resource_id_param=None)
    def post(self, request, version):
        """Create the provided OLT Automation Config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="OLT-AUTO-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            doc_id = get_nested_value(data, ["_id"], None)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT,
                                             details=f"OLT Automation configuration with id {doc_id} already exists")

        return response


# ==================================================
# ======= Global OLT Automation Config View ========
# ==================================================
class GlobalAutomationConfig(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_global_olt_automation_config",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'automation', 'config', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_automation', raise_exception=True))
    def get(self, request, version):
        """Get the Global OLT Automation Config"""
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-AUTO-CFG",
                                             query={"_id": "Global"})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="put_global_olt_automation_config",
        request=inline_serializer(name="Global OLT-AUTO-CFG", fields={"data": JSONField(help_text="OLT-AUTO-CFG")}),
        responses={
            200: ResponseExample(200),
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'automation', 'config', 'put']
    )
    @method_decorator(
        permission_required_any_of(['global_config.can_update_automation', 'global_config.can_create_automation'],
                                   raise_exception=True))
    # TODO @validate_data(collection="OLT-AUTO-CFG", resource_id_param=None)
    def put(self, request, version):
        """Update the Global OLT Automation Config"""
        data = get_nested_value(request.data, ["data"])
        if data is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details="Request body must be of format '{ data: <OLT-AUTO-CFG> }'")
        else:
            old_document = database_manager.find_one_and_replace(database_id=request.session.get('database'),
                                                                 collection="OLT-AUTO-CFG", query={"_id": "Global"},
                                                                 new_document=data)

            if old_document is None:
                response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=data, old_data=old_document)

        return response

    @extend_schema(
        operation_id="post_global_olt_automation_config",
        request=inline_serializer(name="Global OLT-AUTO-CFG", fields={"data": JSONField(help_text="OLT-AUTO-CFG")}),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500)
        },
        tags=['olt', 'automation', 'config', 'post']
    )
    @method_decorator(permission_required('global_config.can_create_automation', raise_exception=True))
    # TODO @validate_data(collection="OLT-AUTO-CFG", resource_id_param=None)
    def post(self, request, version):
        """Create the provided OLT Automation Global Config"""
        try:
            data = get_nested_value(request.data, ["data"])
            database_manager.insert_one(database_id=request.session.get('database'), collection="OLT-AUTO-CFG",
                                        document=data)
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=data, old_data=None)
        except pymongo.errors.DuplicateKeyError:
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT,
                                             details=f"OLT Automation configuration with id Global already exists")

        return response

    @extend_schema(
        operation_id="delete_global_olt_automation_config",
        parameters=[
            OpenApiParameter(name="step", description="Automation step to delete template from",
                             type=OpenApiTypes.STR, required=True),
            OpenApiParameter(name="name", description="Automation template to delete from the specified step",
                             type=OpenApiTypes.STR, required=True)
        ],
        responses={
            204: ResponseExample(204),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'automation', 'config', 'delete']
    )
    @method_decorator(permission_required('global_config.can_delete_automation', raise_exception=True))
    def delete(self, request, version):
        """Delete the Global OLT Automation Config"""
        step = request.GET.get('step', None)
        name = request.GET.get('name', None)

        # Return missing parameter response if step or name are undefined
        if step is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Parameter 'step' is required"})
        elif name is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST,
                                             details={"message": "Parameter 'name' is required"})
        else:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("OLT-AUTO-CFG")
            try:
                if step.upper() == "IDENTIFY":
                    collection.update({'_id': "Global"}, {"$pull": {"IDENTIFY.Mapping": {"Description": name}}})
                else:
                    collection.update({'_id': "Global"}, {"$unset": {f"{step.upper()}.{name}": ""}})
            except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
                raise APIException(detail=f"MongoDB error: {str(e)}")

            response = PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)

        return response


# ==================================================
# ================ OLT Reset View ==================
# ==================================================
class Reset(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_reset_status",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'reset', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """ Check the reset status of the specified OLT """
        database = database_manager.get_database(request.session.get("database"))
        collection = database.get_collection("OLT-STATE")
        try:
            result = list(collection.aggregate([
                {"$match": {"_id": olt_id}},
                {"$lookup": {"from": "OLT-CFG", "localField": "_id", "foreignField": "_id", "as": "CFG"}},
                {"$addFields": {"CFG": {"$arrayElemAt": ["$CFG", 0]}}},
                {"$project": {"_id": 0, "OLT.Reset Count": 1, "CFG.OLT.Reset Count": 1}}
            ]))
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        res_data = {
            "Pending": False
        }
        if len(result) > 0 and result[0]["OLT"] and result[0]["CFG"]:
            # Return pending status as false if State or Config are not found
            state_count = get_nested_value(result, path=[0, "OLT", "Reset Count"], default=0)
            cfg_count = get_nested_value(result, path=[0, "CFG", "OLT", "Reset Count"], default=0)
            res_data["Pending"] = state_count != cfg_count

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="reset_olt",
        request=None,
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            404: ResponseExample(404),
            500: ResponseExample(500),
        },
        tags=['olt', 'reset', 'put']
    )
    @method_decorator(permission_required_any_of(['network.can_update_network_olts', 'network.can_create_network_olts'],
                                                 raise_exception=True))
    def put(self, request, olt_id, version):
        """ Trigger a Reset on the specified OLT """
        update_document = {"$inc": {"OLT.Reset Count": 1, "OLT.CFG Change Count": 1}}
        update_result = database_manager.update_one(database_id=request.session.get("database"), collection="OLT-CFG",
                                                    query={"_id": olt_id}, update_document=update_document)
        if update_result.matched_count == 0:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                             details={"message": f"OLT {olt_id}'s configuration was not found"})
        else:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})

        return response


# ==================================================
# == OLT Allow ONU Registration Schema Definition ==
# ==================================================
ALLOW_ONU_REG_SCHEMA = {
    "properties": {
        "onu-id": {
            "anyOf": [
                {"$ref": "TIBIT-TYPES.json#/definitions/MAC Address"},
                {"$ref": "TIBIT-TYPES.json#/definitions/ONU Serial Number"},
                {"const": "ALL"}
            ]
        }
    }
}


# ==================================================
# ======== OLT Allow ONU Registration View =========
# ==================================================
class AllowRegistration(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_allow_onu_registration",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'registration', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """ Check if the specified ONU can be allowed to register """
        database = database_manager.get_database(request.session.get('database'))
        olt_state_collection = database.get_collection("OLT-STATE")
        try:
            cursor = list(olt_state_collection.aggregate([
                {"$match": {"_id": olt_id}},
                {"$lookup": {"from": "OLT-CFG", "localField": "_id", "foreignField": "_id", "as": "CFG"}},
                {"$addFields": {"CFG": {"$arrayElemAt": ["$CFG", 0]}}},
                {"$project": {"_id": 0, "OLT.Reg Allow ONU": 1, "OLT.Reg Allow Count": 1, "CFG.OLT.Reg Allow ONU": 1,
                              "CFG.OLT.Reg Allow Count": 1}}
            ]))
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        res_data = {
            "Pending": False
        }
        if len(cursor) > 0:
            for olt in cursor:
                state_allow_onu = olt['OLT']['Reg Allow ONU']
                config_allow_onu = olt['CFG']['OLT']['Reg Allow ONU']
                state_allow_count = olt['OLT']['Reg Allow Count']
                config_allow_count = olt['CFG']['OLT']['Reg Allow Count']

                if config_allow_onu != state_allow_onu or config_allow_count != state_allow_count:
                    res_data["Pending"] = True
                    break

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="olt_allow_onu_registration",
        request=RequestSerializer(name="AllowRegistration", data_fields={
            "onu-id": ChoiceField(choices=["<ONU SN/MAC Address>", "'ALL'"])
        }),
        responses={
            200: ResponseExample(200),
            400: ResponseExample(400),
            403: ResponseExample(403),
            404: ResponseExample(404),
            500: ResponseExample(500),
        },
        tags=['olt', 'registration', 'put']
    )
    @method_decorator(permission_required_any_of(['network.can_update_network_olts', 'network.can_create_network_olts'],
                                                 raise_exception=True))
    @validate_data(collection="OLT-CFG", resource_id_param=None, schema=ALLOW_ONU_REG_SCHEMA)
    def put(self, request, data, olt_id, version):
        """ Allow registration for the specified ONU """
        allow_onu = get_nested_value(data, ["onu-id"])
        if allow_onu is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={
                "message": "Request body must be of format '{ data: { onu-id: <ONU MAC Address/SSN> | 'ALL' } }'"})
        else:
            update_document = {"$set": {"OLT.Reg Allow ONU": allow_onu},
                               "$inc": {"OLT.Reg Allow Count": 1, "OLT.CFG Change Count": 1}}
            update_result = database_manager.update_one(database_id=request.session.get("database"),
                                                        collection="OLT-CFG",
                                                        query={"_id": olt_id}, update_document=update_document)
            if update_result.matched_count == 0:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                                 details={"message": f"OLT {olt_id}'s configuration was not found"})
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})

        return response


# ==================================================
# ==== OLT Disable/Enable ONU Schema Definition ====
# ==================================================
DISABLE_ONU_SCHEMA = {
    "properties": {
        "disable": {
            "type": "boolean"
        }
    }
}


# ==================================================
# ========== OLT Disable/Enable ONU View ===========
# ==================================================
class DisableOnu(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_onu_disabled_status",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'disable', 'onu', 'get']
    )
    def get(self, request, olt_id, onu_id, version):
        """ Get the disabled status for the ONU in the OLT's inventory """
        database = database_manager.get_database(request.session.get('database'))
        olt_state_collection = database.get_collection("OLT-STATE")
        try:
            result = list(olt_state_collection.aggregate([
                {"$match": {"_id": olt_id}},
                {"$lookup": {"from": "OLT-CFG", "localField": "_id", "foreignField": "_id", "as": "CFG"}},
                {"$addFields": {"CFG": {"$arrayElemAt": ["$CFG", 0]}}},
                {"$project": {"_id": 0}}
            ]))
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        res_data = {
            "Pending": False
        }
        # Return pending status as false if State or Config are not found
        if len(result) > 0 and result[0]["ONUs"] and result[0]["CFG"]:
            state_disable_count = get_nested_value(result, path=[0, "ONUs", onu_id, "Disable Count"], default=0)
            cfg_disable_count = get_nested_value(result, path=[0, "CFG", "ONUs", onu_id, "Disable Count"], default=0)
            state_enable_count = get_nested_value(result, path=[0, "ONUs", onu_id, "Enable Count"], default=0)
            cfg_enable_count = get_nested_value(result, path=[0, "CFG", "ONUs", onu_id, "Enable Count"], default=0)
            res_data["Pending"] = state_disable_count != cfg_disable_count or state_enable_count != cfg_enable_count

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="olt_disable_onu",
        request=RequestSerializer(name="DisableOnu", data_fields={
            "disable": BooleanField()
        }),
        responses={
            200: ResponseExample(200),
            400: ResponseExample(400),
            403: ResponseExample(403),
            404: ResponseExample(404),
            500: ResponseExample(500),
        },
        tags=['olt', 'disable', 'onu', 'put']
    )
    @method_decorator(permission_required_any_of(['network.can_update_network_olts', 'network.can_create_network_olts'],
                                                 raise_exception=True))
    @validate_data(collection="OLT-CFG", resource_id_param=None, schema=DISABLE_ONU_SCHEMA)
    def put(self, request, data, olt_id, onu_id, version):
        """ Update the disable/enable signal for the ONU in the OLT's inventory """
        disable = get_nested_value(data, ["disable"])
        if disable is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details={
                "message": "Request body must be of format '{ data: { disable: <bool> } }'"})
        else:
            if disable:
                update_document = {"$set": {f"ONUs.{onu_id}.Disable": True},
                                   "$inc": {f"ONUs.{onu_id}.Disable Count": 1, "OLT.CFG Change Count": 1}}
            else:
                update_document = {"$set": {f"ONUs.{onu_id}.Disable": False},
                                   "$inc": {f"ONUs.{onu_id}.Enable Count": 1, "OLT.CFG Change Count": 1}}

            update_result = database_manager.update_one(database_id=request.session.get("database"),
                                                        collection="OLT-CFG",
                                                        query={"_id": olt_id}, update_document=update_document)
            if update_result.matched_count == 0:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                                 details={"message": f"OLT {olt_id}'s configuration was not found"})
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})

        return response


# ==================================================
# ========= OLT Broadcast Enable ONUs View =========
# ==================================================
class BroadcastEnableOnus(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_broadcast_enable_status",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'enable', 'broadcast', 'onu', 'get']
    )
    def get(self, request, olt_id, version):
        """ Get the status of the OLTs broadcast enable signal """
        database = database_manager.get_database(request.session.get('database'))
        olt_state_collection = database.get_collection("OLT-STATE")
        try:
            result = list(olt_state_collection.aggregate([
                {"$match": {"_id": olt_id}},
                {"$lookup": {"from": "OLT-CFG", "localField": "_id", "foreignField": "_id", "as": "CFG"}},
                {"$addFields": {"CFG": {"$arrayElemAt": ["$CFG", 0]}}},
                {"$project": {"_id": 0}}
            ]))
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        res_data = {
            "Pending": False
        }
        if len(result) > 0 and result[0]["ONUs"] and result[0]["CFG"]:
            # Return pending status as false if State or Config are not found
            state_count = get_nested_value(result, path=[0, "ONU", "Enable All Serial Numbers Count"], default=0)
            cfg_count = get_nested_value(result, path=[0, "CFG", "ONU", "Enable All Serial Numbers Count"], default=0)
            res_data["Pending"] = state_count != cfg_count

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="olt_broadcast_enable_onus",
        request=None,
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            404: ResponseExample(404),
            500: ResponseExample(500),
        },
        tags=['olt', 'enable', 'broadcast', 'onu', 'put']
    )
    @method_decorator(permission_required_any_of(['network.can_update_network_olts', 'network.can_create_network_olts'],
                                                 raise_exception=True))
    def put(self, request, olt_id, version):
        """ Update the OLT's enable all ONUs count """
        update_document = {"$inc": {"ONU.Enable All Serial Numbers Count": 1, "OLT.CFG Change Count": 1}}
        update_result = database_manager.update_one(database_id=request.session.get("database"), collection="OLT-CFG",
                                                    query={"_id": olt_id}, update_document=update_document)
        if update_result.matched_count == 0:
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND,
                                             details={"message": f"OLT {olt_id}'s configuration was not found"})
        else:
            response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})

        return response


# ==================================================
# ======== OLT Protection Sync OLTs View =========
# ==================================================
class ProtectionSyncOlts(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_olt_protection_sync_olts",
        exclude=True,
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['olt', 'protection', 'sync', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_olts', raise_exception=True))
    def get(self, request, olt_id, version):
        """ Get the Protection Peer for the specified OLT """
        res_data = database_manager.find_one(database_id=request.session.get('database'), collection="OLT-CFG",
                                             query={"_id": olt_id}, projection={"_id": 0, "Protection.Peer": 1})

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)

    @extend_schema(
        operation_id="olt_protection_sync_olts",
        request=None,
        exclude=True,
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            404: ResponseExample(404),
            500: ResponseExample(500),
        },
        tags=['olt', 'protection', 'sync', 'put']
    )
    @method_decorator(permission_required_any_of(['network.can_update_network_olts', 'network.can_create_network_olts'],
                                                 raise_exception=True))
    def put(self, request, olt_id, version):
        """ Apply specified OLTs config to the configured Protection Peer OLT """

        # Find the specified OLTs protection peers OLT-CFG document, and add it to results
        database = database_manager.get_database(request.session.get("database"))
        collection = database.get_collection("OLT-CFG")
        try:
            olt_cfg_peer_aggregate = list(collection.aggregate([
                {"$match": {"_id": olt_id}},
                {"$lookup": {"from": "OLT-CFG", "localField": "Protection.Peer", "foreignField": "_id",
                             "as": "PEER_CFG"}},
                {"$lookup": {"from": "OLT-STATE", "localField": "Protection.Peer", "foreignField": "_id",
                             "as": "PEER_STATE"}},
                {"$addFields": {"PEER_CFG": {"$arrayElemAt": ["$PEER_CFG", 0]}}},
                {"$addFields": {"PEER_STATE": {"$arrayElemAt": ["$PEER_STATE", 0]}}}
            ]))[0]
        except (ConnectionRefusedError, pymongo.errors.PyMongoError) as e:
            raise APIException(detail=f"MongoDB error: {str(e)}")

        if 'PEER_CFG' not in olt_cfg_peer_aggregate:
            # The 'PEER_CFG' property will be missing in a few scenarios:
            # If the configuration file for the specified OLTs protection peer was not found. Expected behavior should result in error.
            # If the OLT does not have a protection peer configured. Expected behavior should result in success.
            # If the OLT CFG is an old document not containing the Protection.Peer field. Expected behavior should result in success.
            response = PonManagerApiResponse(status=status.HTTP_200_OK)
        else:
            ### Syncing ONUs object ###
            # Loop through ONUs that should be removed from peer
            onus_inventoried_on_source_olt = copy.deepcopy(olt_cfg_peer_aggregate['ONUs']).keys()
            onus_inventoried_on_peer_olt = copy.deepcopy(olt_cfg_peer_aggregate['PEER_CFG']['ONUs']).keys()
            for onu in onus_inventoried_on_peer_olt:
                if onu not in onus_inventoried_on_source_olt:
                    del olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]

            # Loop through ONUs that should exist in peer
            for onu in olt_cfg_peer_aggregate['ONUs']:
                # If the ONU doesn't exist in the peer, we must add it
                if onu not in olt_cfg_peer_aggregate['PEER_CFG']['ONUs']:
                    # Copy ONU inventory record from source OLT to peer OLT
                    olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu] = olt_cfg_peer_aggregate['ONUs'][onu]

                    # If the Peer OLT has enable/disable counts in state, use those
                    if 'PEER_STATE' in olt_cfg_peer_aggregate and onu in olt_cfg_peer_aggregate['PEER_STATE']['ONUs']:
                        if 'Enable Count' in olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]:
                            olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Enable Count'] = \
                            olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]['Enable Count']
                        else:
                            olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Enable Count'] = 0
                        if 'Disable Count' in olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]:
                            olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Disable Count'] = \
                            olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]['Disable Count']
                        else:
                            olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Disable Count'] = 0
                    # If not, if not, write zeros for counts
                    else:
                        olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Enable Count'] = 0
                        olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Disable Count'] = 0

                # Peer OLT has ONU in inventory. We must still update everything except counts
                else:
                    # Setting Enable Count
                    if 'Enable Count' in olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]:
                        current_enable_count = olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Enable Count']
                    elif 'PEER_STATE' in olt_cfg_peer_aggregate and onu in olt_cfg_peer_aggregate['PEER_STATE'][
                        'ONUs'] and 'Enable Count' in olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]:
                        current_enable_count = olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]['Enable Count']
                    else:
                        current_enable_count = 0

                    if 'Disable Count' in olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]:
                        current_disable_count = olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Disable Count']
                    elif 'PEER_STATE' in olt_cfg_peer_aggregate and onu in olt_cfg_peer_aggregate['PEER_STATE'][
                        'ONUs'] and 'Disable Count' in olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]:
                        current_disable_count = olt_cfg_peer_aggregate['PEER_STATE']['ONUs'][onu]['Disable Count']
                    else:
                        current_disable_count = 0

                    olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu] = olt_cfg_peer_aggregate['ONUs'][onu]
                    olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Enable Count'] = current_enable_count
                    olt_cfg_peer_aggregate['PEER_CFG']['ONUs'][onu]['Disable Count'] = current_disable_count

            ### Updating OLT peer configuration ###
            update_document = {
                "$set": {"EPON": olt_cfg_peer_aggregate['EPON'],
                         "GPON.Discovery Period": olt_cfg_peer_aggregate['GPON']['Discovery Period'],
                         "GPON.Downstream Fec": olt_cfg_peer_aggregate['GPON']['Downstream Fec'],
                         "GPON.Encryption": olt_cfg_peer_aggregate['GPON']['Encryption'],
                         "GPON.Error Det Min Sample": olt_cfg_peer_aggregate['GPON']['Error Det Min Sample'],
                         "GPON.Error Det Max Ratio": olt_cfg_peer_aggregate['GPON']['Error Det Max Ratio'],
                         "GPON.Guard Time": olt_cfg_peer_aggregate['GPON']['Guard Time'],
                         "GPON.Max Frame Size": olt_cfg_peer_aggregate['GPON']['Max Frame Size'],
                         "GPON.Upstream FEC 0": olt_cfg_peer_aggregate['GPON']['Upstream FEC 0'],
                         "GPON.Upstream FEC 1": olt_cfg_peer_aggregate['GPON']['Upstream FEC 1'],
                         "GPON.Upstream FEC 2": olt_cfg_peer_aggregate['GPON']['Upstream FEC 2'],
                         "GPON.Upstream FEC 3": olt_cfg_peer_aggregate['GPON']['Upstream FEC 3'],
                         "GPON.Upstream Preamble 0": olt_cfg_peer_aggregate['GPON']['Upstream Preamble 0'],
                         "GPON.Upstream Preamble 1": olt_cfg_peer_aggregate['GPON']['Upstream Preamble 1'],
                         "GPON.Upstream Preamble 2": olt_cfg_peer_aggregate['GPON']['Upstream Preamble 2'],
                         "GPON.Upstream Preamble 3": olt_cfg_peer_aggregate['GPON']['Upstream Preamble 3'],

                         "NNI Networks": olt_cfg_peer_aggregate['NNI Networks'],
                         "OLT.PON Enable": olt_cfg_peer_aggregate['OLT']['PON Enable'],
                         "OLT.PON Mode": olt_cfg_peer_aggregate['OLT']['PON Mode'],
                         "OLT.Max Round Trip Time": olt_cfg_peer_aggregate['OLT']['Max Round Trip Time'],
                         "ONU.Max FW Upgrades": olt_cfg_peer_aggregate['ONU']['Max FW Upgrades'],
                         "ONUs": olt_cfg_peer_aggregate['PEER_CFG']['ONUs'],
                         "Protection.Inactive Periods.Active": olt_cfg_peer_aggregate['Protection']['Inactive Periods'][
                             'Active'],
                         "Protection.Inactive Periods.Standby":
                             olt_cfg_peer_aggregate['Protection']['Inactive Periods']['Standby'],
                         "MAC Learning": olt_cfg_peer_aggregate['MAC Learning'],
                         },
                "$inc": {
                    "OLT.CFG Change Count": 1
                }
            }
            update_result = database_manager.update_one(database_id=request.session.get("database"),
                                                        collection="OLT-CFG",
                                                        query={"_id": olt_cfg_peer_aggregate['Protection']['Peer']},
                                                        update_document=update_document)
            if update_result.matched_count == 0:
                response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND, details={
                    "message": f"Peer OLT {olt_id}'s configuration was not found"})
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=update_document, old_data={})

        return response
