"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import copy
import sys
import traceback

from django.contrib.auth.decorators import permission_required
from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from pymongo.errors import ConfigurationError, DuplicateKeyError
from rest_framework.fields import BooleanField, CharField, ListField
from rest_framework.generics import GenericAPIView
from rest_framework.views import APIView
from drf_spectacular.utils import extend_schema
from rest_framework import status

import DBdjango

from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.serializers import CharFieldSerializer, RequestSerializer
from utils.tools import PonManagerApiResponse, get_nested_value, validate_data, permission_required_any_of


# ==================================================
# ============= Databases Status View ==============
# ==================================================
class ConnectionStatus(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_database_statuses",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['database', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_databases', raise_exception=True))
    def get(self, request, version):
        """Get the alive status for all databases"""
        if version == "v1":
            # v1 code as of R3.0.0
            all_databases = database_manager.get_all_databases()
            res_data = {}
            for key in all_databases.keys():
                res_data[key] = all_databases[key][1].is_alive
        else:
            # v2 code as of R3.1.0
            all_databases = database_manager.get_all_databases()
            res_data = {}

            for key in all_databases.keys():
                res_data[key] = {"is_alive": all_databases[key][1].is_alive, "details": database_manager.mongo_server_get_status(key)}

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ============ One Database Status View ============
# ==================================================
class OneConnectionStatus(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_database_status",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['database', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_databases', raise_exception=True))
    def get(self, request, database_id, version):
        """Get the alive status for the given database"""
        if version == "v1":
            # v1 code as of R3.0.0
            try:
                res_data = {"is_alive": database_manager.mongo_server_is_active(database_id)}
            except KeyError:
                res_data = {"is_alive": False}
        else:
            # v2 code as of R3.1.0
            try:
                res_data = {"is_alive": database_manager.mongo_server_is_active(database_id), "details": database_manager.mongo_server_get_status(database_id)}
            except KeyError:
                res_data = {"is_alive": False, "status": "Not found"}

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ============= Manage Databases View ==============
# ==================================================
class Manage(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_databases",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['database', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_databases', raise_exception=True))
    def get(self, request, version):
        """Get the connection parameters for all active databases"""
        res_data = database_manager.get_databases_json()

        return PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)


# ==================================================
# ============== Select Database View ==============
# ==================================================
class Select(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_selected_database",
        responses={
            200: ResponseExample(200),
            400: ResponseExample(400),
            500: ResponseExample(500),
        },
        tags=['database', 'get']
    )
    def get(self, request, version):
        """Get the users active database selection"""
        try:
            database_id = database_manager.get_users_selected_database(request.user.email)
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=database_id)
        except KeyError:
            traceback.print_exc(file=sys.stdout)
            response = PonManagerApiResponse(status=status.HTTP_500_INTERNAL_SERVER_ERROR, details="User does not have a selected database ID or the database ID does not exist")

        return response

    @extend_schema(
        operation_id="select_database",
        request=CharFieldSerializer,
        responses={
            200: ResponseExample(200),
            400: ResponseExample(400),
            500: ResponseExample(500),
        },
        tags=['database', 'put']
    )
    def put(self, request, version):
        """Change the users active database selection"""
        database_key = get_nested_value(request.data, ['data'])
        if database_key is None or not isinstance(database_key, str):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Request body must be of format '{ data: <Database ID> }'")
        else:
            update_result = database_manager.set_users_selected_database(user_email=request.user.email, database_id=database_key)
            old_database_id = database_manager.get_session_database_id(request.session.session_key)
            database_manager.set_session_database_id(session_key=request.session.session_key, database_id=database_key)
            DBdjango.views.clients[request.user.email] = database_key

            # User was not found. Does not include message as to reason for error to avoid giving information on existing users
            if update_result.matched_count == 0:
                response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST)
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=database_key, old_data=old_database_id)

        return response


# ==================================================
# =========== SessionOnly Database View ============
# ==================================================
class SessionOnly(LoginRequiredMixin, APIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_selected_database_by_session",
        responses={
            200: ResponseExample(200),
            400: ResponseExample(400),
            500: ResponseExample(500),
        },
        tags=['database', 'get']
    )
    def get(self, request, version):
        """Get the active database selection for the session"""
        try:
            database_id = database_manager.get_session_database_id(request.session.session_key)
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=database_id)
        except KeyError:
            traceback.print_exc(file=sys.stdout)
            response = PonManagerApiResponse(status=status.HTTP_500_INTERNAL_SERVER_ERROR, details="User does not have a selected database ID or the database ID does not exist")

        return response

    @extend_schema(
        operation_id="select_database_for_session",
        request=CharFieldSerializer,
        responses={
            200: ResponseExample(200),
            400: ResponseExample(400),
            500: ResponseExample(500),
        },
        tags=['database', 'put']
    )
    def put(self, request, version):
        """Change the active database selection for the session"""
        database_key = get_nested_value(request.data, ['data'])
        if database_key is None or not isinstance(database_key, str):
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details="Request body must be of format '{ data: <Database ID> }'")
        else:
            old_database_id = database_manager.get_session_database_id(request.session.session_key)
            database_manager.set_session_database_id(request.session.session_key, database_key)
            request.session.update({"database": database_key})
            response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=database_key, old_data=old_database_id)

        return response


# ==================================================
# ==== Manage One Database Schema Definition =======
# ==================================================
MANAGE_DATABASE_SCHEMA = {
    "properties": {
        "auth_db": {
            "type": "string",
            "pattern": "^([^\/\\. \';$*<>:|?]+)$",
            "maxlength": 63
        },
        "auth_enable": {
            "type": "boolean",
        },
        "ca_cert_path": {
            "type": "string"
        },
        "db_uri": {
            "type": "string"
        },
        "dns_srv": {
            "type": "boolean"
        },
        "host": {
            "type": "string",
            "anyOf" : [
                    { "format": "hostname" },
                    { "format": "ipv4" },
                    { "format": "ipv6" },
                    { "const": "" }
            ],
        },
        "name": {
            "type": "string",
            "pattern": "^([^\/\\. \';$*<>:|?]+)$",
            "maxlength": 63
        },
        "password": {
            "type": "string"
        },
        # TCP Port Number (0 - 65535)
        "port": {
            "type": "string",
            "pattern": "^([0-9]|[1-9][0-9]{1,3}|[1-5][0-9]{4}|6[0-4][0-9]{3}|65[0-4][0-9]{2}|655[0-2][0-9]|6553[0-5])$"
        },
        "replica_set_enable": {
            "type": "boolean"
        },
        "replica_set_hosts": {
            "type": "array",
            "items": {
                "type": "string"
            }
        },
        "replica_set_name": {
            "type": "string"
        },
        "tls_enable": {
            "type": "boolean"
        },
        "username": {
            "type": "string"
        }
    }
}


# ==================================================
# ============ Manage One Database View ============
# ==================================================
class ManageOne(LoginRequiredMixin, GenericAPIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        operation_id="get_one_database",
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            404: ResponseExample(404),
            500: ResponseExample(500),
        },
        tags=['database', 'get']
    )
    @method_decorator(permission_required('global_config.can_read_global_config_databases', raise_exception=True))
    def get(self, request, database_id, version):
        """Get the connection parameters for the specified database"""
        try:
            res_data = database_manager.get_databases_json()[database_id]
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=res_data)
        except KeyError as e:
            traceback.print_exc(file=sys.stdout)
            response = PonManagerApiResponse(status=status.HTTP_404_NOT_FOUND)

        return response

    @extend_schema(
        operation_id="put_one_database",
        request=RequestSerializer(name="UpdateDatabaseInfo", data_fields={
            "auth_db": CharField(),
            "auth_enable": BooleanField(),
            "ca_cert_path": CharField(),
            "db_uri": CharField(),
            "dns_srv": BooleanField(),
            "host": CharField(),
            "name": CharField(),
            "password": CharField(),
            "port": CharField(),
            "replica_set_enable": BooleanField(),
            "replica_set_hosts": ListField(),
            "replica_set_name": CharField(),
            "tls_enable": BooleanField(),
            "username": CharField()
        }),
        responses={
            200: ResponseExample(200),
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['database', 'put']
    )
    @method_decorator(permission_required_any_of(['global_config.can_update_global_config_databases', 'global_config.can_create_global_config_databases'], raise_exception=True))
    @validate_data(collection="DATABASES", resource_id_param=None, schema=MANAGE_DATABASE_SCHEMA)
    def put(self, request, data, database_id, version):
        """Update the connection parameters for the specified database"""
        try:
            old_data = None
            databases_list = database_manager.get_all_databases()
            if database_id in databases_list.keys():
                old_data = database_manager.get_databases_json()[database_id]
                old_data["password"] = "*****"

            edit_result = database_manager.edit_database(database_id, data)
            DBdjango.views.start_up()

            no_password_new_data = copy.deepcopy(data)
            no_password_new_data["password"] = "*****"

            if edit_result == "Created":
                response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=no_password_new_data, old_data=old_data)
            else:
                response = PonManagerApiResponse(status=status.HTTP_200_OK, new_data=no_password_new_data, old_data=old_data)
        except (ConfigurationError, KeyError) as e:
            traceback.print_exc(file=sys.stdout)
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details=str(e))

        return response

    @extend_schema(
        operation_id="post_one_database",
        request=RequestSerializer(name="CreateDatabaseInfo", data_fields={
            "auth_db": CharField(),
            "auth_enable": BooleanField(),
            "ca_cert_path": CharField(),
            "db_uri": CharField(),
            "dns_srv": BooleanField(),
            "host": CharField(),
            "name": CharField(),
            "password": CharField(),
            "port": CharField(),
            "replica_set_enable": BooleanField(),
            "replica_set_hosts": ListField(),
            "replica_set_name": CharField(),
            "tls_enable": BooleanField(),
            "username": CharField()
        }),
        responses={
            201: ResponseExample(201),
            400: ResponseExample(400),
            403: ResponseExample(403),
            409: ResponseExample(409),
            500: ResponseExample(500),
        },
        tags=['database', 'post']
    )
    @method_decorator(permission_required('global_config.can_create_global_config_databases', raise_exception=True))
    @validate_data(collection="DATABASES", resource_id_param=None, schema=MANAGE_DATABASE_SCHEMA)
    def post(self, request, data, database_id, version):
        """Create a new connection to the given database"""
        try:
            database_manager.add_database(database_id, data)
            DBdjango.views.start_up()
            no_password_new_data = copy.deepcopy(data)
            no_password_new_data["password"] = "*****"
            response = PonManagerApiResponse(status=status.HTTP_201_CREATED, new_data=no_password_new_data)
        except DuplicateKeyError as e:
            traceback.print_exc(file=sys.stdout)
            response = PonManagerApiResponse(status=status.HTTP_409_CONFLICT, details="All database IDs must be unique.")
        except (ConfigurationError, KeyError) as e:
            traceback.print_exc(file=sys.stdout)
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, details=str(e))

        return response

    @extend_schema(
        operation_id="delete_one_database",
        responses={
            204: ResponseExample(204),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['database', 'delete']
    )
    @method_decorator(permission_required('global_config.can_delete_global_config_databases', raise_exception=True))
    def delete(self, request, database_id, version):
        """Delete the given database connection"""
        database_manager.remove_database(database_id)
        DBdjango.views.start_up()

        return PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
